/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <setjmp.h>
#ifdef HAVE_LIBREADLINE
#  include <readline/tilde.h>
#endif

#include "cmp.h"
#include "kalc.h"


int dirty;

void f_save(void) 
{
  /*
   * This is the function called by the user command "save". It saves
   * the status of the calculator in the current file.
   */

  if (!saveStatusToFile(currStatusFile))
    fputs("Could not write status file\n", stderr);
}


void f_saveAs(void) 
{
  /*
   * This tries to save the status to the file named in level one.
   * The current status file variable is updated if the opearation is
   * successful.
   */

  char tmp[PATHSIZE];
  char *fileName;

  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_ID) {
#ifdef HAVE_LIBREADLINE
      fileName = tilde_expand((**tos).value.str);
#else
      fileName = (**tos).value.str;
#endif
      strcpy(tmp, fileName);
#ifdef HAVE_LIBREADLINE
      free(fileName);
#endif
      _f_drop();
      if (saveStatusToFile(tmp))
	updateCurrentStatusFile(tmp);
      else
	doError("saveAs", ERR_COULDNOTOPENFILE);
    } else
      doError("saveAs", ERR_BADARGUMENTTYPE);
  } else
    doError("saveAs", ERR_TOOFEWARGUMENTS);
}


void f_open(void) 
{
  /*
   * This function tries to load the status from the specified file.
   */

  char *fileName;
  
  if (enoughArgs(1)) {
    if (type(**tos) == TYPE_ID) {
      if (!dirty || userOptions.expert || wantsToContinue()) {	
#ifdef HAVE_LIBREADLINE
	fileName = tilde_expand((**tos).value.str);
#else
	fileName = (**tos).value.str;
#endif

	f_coldStart();
	readStatusFromFile(fileName, 1, 1);
	updateCurrentStatusFile(fileName);
	
#ifdef HAVE_LIBREADLINE
	free(fileName);
#endif
      }
    } else
      doError("open", ERR_BADARGUMENTTYPE);
  } else
    doError("open", ERR_TOOFEWARGUMENTS);
}


void f_coldStart(void) 
{
  /*
   * This function clears the stack, the memory, the last args,
   * and resets options to their default values.
   * The user can call it, but there aren't many uses for that. But this
   * function is called by the f_load() function.
   */

  f_clear();
  _doClvar();
  saveLastArgs(0);
  setDefaults();
}


/**********************************************************
 * End of the user functions                              *
 **********************************************************/
int wantsToContinue(void) 
{
  /*
   * This function asks the user wheter he wants to continue the
   * action (opening a file) if the file is dirty, saving it or no
   * before.
   *
   * Returns 1 if the action is to be continued, or 0 if it is to be
   * cancelled.
   */

  char buffer[10];
  char answer;
  
  printf("The status file has not been saved. Do you want to save it?\n"
	 "(Y)es, (N)o, (C)ancel: ");
  fgets(buffer, sizeof(buffer), stdin);
  answer = toupper(*buffer);
  while (answer != 'Y' && answer != 'N' && answer != 'C') {
    printf("Please answer Y, N or C: ");
    fgets(buffer, sizeof(buffer), stdin);
    answer = toupper(*buffer);
  }

  if (answer == 'Y')
    saveStatusToFile(currStatusFile);

  if (answer != 'C')
    return 1;

  return 0;
}   
  

int readStatusFromFile(char *file, int loadStack, int loadMem)
{
  /*
   * This function tries to read the calculator status from the
   * given file.
   *
   * Returns 1 on success, 0 on failure.
   */
  
  FILE *fp;
  int status = 0;
  
  if ((fp = fopen(file, "rb")) != NULL) {
    char ver[8];

    /* Get version */
    fread(ver, 1, 8, fp);
    if (!strcmp(ver, MAGICVER) || !strcmp(ver, MAGICVER210))
      status = readStatus(fp, loadStack, loadMem);
    else if (!strcmp(ver, MAGICVER200))
      status = readStatus200(fp, loadStack, loadMem);
    else
      return 0;
    
    fclose(fp);
    dirty = 0;
  }

  return status;
}


int saveStatusToFile(char *file)
{
  /*
   * This function opens the specified file and saves the calculator
   * status to it.
   *
   * Returns 1 on success, 0 on failure.
   */
  
  FILE *fp;

  if ((fp = fopen(file, "wb")) != NULL) {
    saveStatus(fp);
    fclose(fp);
    dirty = 0;
    return 1;
  }

  return 0;
}


int readStatus(FILE *fp, int loadStack, int loadMem) 
{
  /*
   * This functions tries to read a saved status from the calculator
   * from the file pointed to by fp.
   *
   * Returns 1 on success, 0 on failure.
   *
   * For information on the format of the status file, see the
   * saveStatus() function.
   */
   
  int nStack;
  register int i;
  Object el;

  /* Get Options */
  readUserOptions(fp);
    
  /* Load last args */
  loadLastArgs(fp);

  /* Get number of stack elements */
  fread(&nStack, sizeof(int), 1, fp);

  /* Get each element and insert it in the stack if desired
     NOTE: We must load all objects because the memory is stored in
     sequence. */
  for (i = 0; i < nStack; i++) {
    el = loadObject(fp);
    if (loadStack) {
      /*
      insertObject(el);
      freeObjectSpecials(el);
      */
      insertNoDup(el);
    }
  }

  /* Load memory contents if asked */
  if (loadMem)
    loadMemory(fp);

  return 1;
}


void saveStatus(FILE *fp) 
{
  /*
   * This function saves the calculator status in the file pointed to by
   * fp.
   *
   * The status is saved this way:
   * - First, eight bytes contain a magic string that identifies the
   *   version.
   * - Then, the options are stored with the storeUserOptions() function
   *   (actually, a macro).
   * - After that, the last arguments, as stored by the storeLastArgs()
   *   function.
   * - Next the stack is saved with the saveStack() function.
   * - Finally, the memory is stored with the saveMemory() function.
   */
  
  /* Write version information */
  fwrite(MAGICVER, 1, 8, fp);

  /* Write user options */
  storeUserOptions(fp);
  
  /* Write last args */
  storeLastArgs(fp);

  /* Write stack elements */
  saveStack(fp);
  
  /* Save memory contents */
  saveMemory(fp);
}


void updateCurrentStatusFile(char *file) 
{
  /*
   * This function updates the current status file information so
   * that it now contains file. If file is a relative path (that is,
   * it's first character is not '/'), the current path is prepended
   * to it.
   * On MS-DOS systems, it is more difficult to determine if a file path
   * is absolute or not, because it may contain a drive letter (and
   * then be an absolute path which does start with a character
   * different from a slash. Also, forward slashes must be considered.
   */
  
#ifdef MSDOS
  if (*file != '/' || *file != '\\' || file[1] != ':') {
#else /* !MSDOS */
  if (*file != '/') {
#endif /* !MSDOS */

#ifdef HAVE_GETCWD
    getcwd(currStatusFile, PATHSIZE);
#  ifdef MSDOS
    strcat(currStatusFile, "\\");
#  else /* !MSDOS */
    strcat(currStatusFile, "/");
#  endif /* !MSDOS */
    strcat(currStatusFile, file);
#else /* !HAVE_GETCWD */
    strcpy(currStatusFile, file);
#endif /* !HAVE_GETCWD */
    
  } else
    strcpy(currStatusFile, file);
}


void saveStack(FILE *fp) 
{
  /*
   * This function saves the stack to the file pointed to by fp.
   *
   * The stack is stored this way:
   * - First, an int with the number of stack elements is stored.
   * - Then, each object is stored with the saveObject() function.
   */

  int nStack;
  register int i;
  
  nStack = _f_depth();
  fwrite(&nStack, sizeof(int), 1, fp);
  for (i = nStack; i > 0; i--)
    saveObject(*(tos - i + 1), fp);
}


void saveObject(Object *obj, FILE *fp)
{
  /*
   * This function saves an object to the file whose pointer if fp.
   *
   * Objects are stored this way:
   * - An int is stored, representing the object's type.
   * - Then, different actions for the different types of objects:
   *   . Real and complex numbers are written directly.
   *   . Hex strings are also written directly.
   *   . Strings and identifiers contents are stored via the saveString()
   *     function.
   *   . Functions have their names stored via saveString()
   *   . For tagged objects, first the tag is stored via saveString(),
   *     then the object is saved by means of a recursive call to this
   *     function.
   *   . For programs, each object is stored sequencially by means of
   *     recursive calls to this function. An object whose type is
   *     NOTANOBJECT finishes the program.
   */

  fwrite(&obj->type, sizeof(int), 1, fp);
  switch (type(*obj)) {
  case TYPE_REAL:
    fwrite(&obj->value.real, sizeof(double), 1, fp);
    break;

  case TYPE_CMP:
    fwrite(&obj->value.cmp, sizeof(Complex), 1, fp);
    break;

  case TYPE_HXS:
    fwrite(&obj->value.h, sizeof(hxs), 1, fp);
    break;

  case TYPE_STR:
  case TYPE_ID:
  case TYPE_UNQUOTEDID:
    saveString(obj->value.str, fp);
    break;

  case TYPE_FUNC:
    saveString(getFuncName(obj->value.func), fp);
    break;

  case TYPE_TAGGED:
    saveString(obj->value.tagged.tag, fp);
    saveObject(obj->value.tagged.obj, fp);
    break;

  case TYPE_PROG:
    saveProgram(obj->value.comp, fp);
    break;
  }
}


Object loadObject(FILE *fp)
{
  /*
   * This option reads an object from the file whose pointer if fp,
   * returning the object. The caller must assure there is an object to
   * be read, otherwise unpredictable results may happen.
   *
   * For information on how the objects are stored, see the saveObject
   * function.
   */

  Object obj, objtmp;
  int type;
  char *tmp;

  fread(&type, sizeof(int), 1, fp);
  obj.type = type;

  switch (type) {
  case TYPE_REAL:
    fread(&obj.value.real, sizeof(double), 1, fp);
    break;

  case TYPE_CMP:
    fread(&obj.value.cmp, sizeof(Complex), 1, fp);
    break;

  case TYPE_HXS:
    fread(&obj.value.h, sizeof(hxs), 1, fp);
    break;

  case TYPE_STR:
  case TYPE_ID:
  case TYPE_UNQUOTEDID:
    obj.value.str = loadString(fp);
    break;

  case TYPE_FUNC:
    tmp = loadString(fp);
    obj.value.func = getFuncAddress(tmp);
    free(tmp);
    break;

  case TYPE_TAGGED:
    obj.value.tagged.tag = loadString(fp);
    objtmp = loadObject(fp);
    obj.value.tagged.obj = objdup(&objtmp);
    freeObjectSpecials(objtmp);
    break;

  case TYPE_PROG:
    obj.value.comp = loadProgram(fp);
    break;
  }

  return obj;
}


void saveString(const char *str, FILE *fp) 
{
  /*
   * This function stores a string in the file pointed to by fp.
   * First, a size_t representing the string's length is written. Then,
   * the characters are written, without a NUL terminator.
   */

  size_t length = strlen(str);

  fwrite(&length, sizeof(length), 1, fp);
  fwrite(str, 1, length, fp);
}


char *loadString(FILE *fp)
{
  /*
   * This function reads a string (stored by the saveString() function)
   * from the file pointed to by fp.
   * The result is malloc()'ed.
   */

  size_t length;
  char *str;

  fread(&length, sizeof(length), 1, fp);

  str = (char *) calloc(length + 1, 1);
  if (!str)
    doError("Fatal", ERR_NOTENOUGHMEMORY);

  fread(str, 1, length, fp);

  return str;
}


void
saveProgram(Composite *comp, FILE *fp)
{
  /*
   * This function saves a program, starting at the given object.
   */

  int NAO = TYPE_NOTANOBJECT;

  while (comp) {
    saveObject(comp->obj, fp);
    comp = comp->next;
  }

  fwrite(&NAO, sizeof(int), 1, fp);
}


Composite *
loadProgram(FILE *fp)
{
  /*
   * This function loads a program from the given file.
   */

  Composite *comp = NULL, *firstComp = NULL, *lastComp = NULL;
  Object *obj;

  while (1) {
    obj = (Object *) malloc(sizeof(Object));
    if (!obj)
      doError("Fatal", ERR_NOTENOUGHMEMORY);

    *obj = loadObject(fp);
    if (type(*obj) == TYPE_NOTANOBJECT)
      break;

    comp = (Composite *) malloc(sizeof(Composite));
    if (!comp)
      doError("Fatal", ERR_NOTENOUGHMEMORY);
    
    if (!firstComp)
      firstComp = comp;
    
    comp->prev = lastComp;
    if (lastComp)
      lastComp->next = comp;

    comp->obj = obj;
    lastComp = comp;
  }

  if (comp)
    comp->next = NULL;

  return firstComp;
}
