/*
     kalc: A Scientific RPN Calculator
     Copyright (C) 1999-2000 Eduardo M Kalinowski (ekalin@iname.com)

     This program is free software. You may redistribute it, but only in
     its whole, unmodified form. You are allowed to make changes to this
     program, but you must not redistribute the changed version.

     This program is distributed in the hope it will be useful, but there
     is no warranty.

     For details, see the COPYING file.
*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <setjmp.h>

#include "cmp.h"
#include "kalc.h"
#include "cmd_time.h"


static char *wdays[] = { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };


void f_date(void)
{
  /*
   * This function inserts the current date in the stack, in the format
   * MM.DDYYYY
   */

  double theDate;
  time_t currTime;
  struct tm *timeinfo;

  currTime = time(NULL);
  timeinfo = localtime(&currTime);

  theDate = __createMDY(timeinfo->tm_mon + 1,
		       timeinfo->tm_mday,
		       timeinfo->tm_year + 1900);

  insertReal(theDate);
}


void f_time(void)
{
  /*
   * This function puts the current time in the stack, in the format
   * HH.MMSS
   */

  double theTime;
  time_t currTime;
  struct tm *timeinfo;

  currTime = time(NULL);
  timeinfo = localtime(&currTime);

  theTime = (timeinfo->tm_hour
	     + timeinfo->tm_min*1e-2
	     + timeinfo->tm_sec*1e-4);

  insertReal(theTime);
}


void f_TOhms(void)
{
  /*
   * This function calls the _f_TOhms function through the wrapper.
   */
  
  run1_1_Function(_f_TOhms, ">hms");
}


void f_hmsTO(void)
{
  /*
   * This function calls the _f_hmsTO function through the wrapper.
   */

  run1_1_Function(_f_hmsTO, "hms>");
}


void f_dateAdd(void)
{
  /*
   * This function calls the _f_dateAdd function through the wrapper.
   */

  run2_1_Function(_f_dateAdd, "date+");
}


void f_ddays(void)
{
  /*
   * This function calls the _f_ddays function through the wrapper.
   */

  run2_1_Function(_f_ddays, "ddays");
}


void f_hmsAdd(void)
{
  /*
   * This function calls the _f_hmsAdd function through the wrapper.
   */

  run2_1_Function(_f_hmsAdd, "hms+");
}


void f_hmsSub(void)
{
  /*
   * This function calls the _f_hmsSub function through the wrapper.
   */

  run2_1_Function(_f_hmsSub, "hms-");
}


void f_dow(void)
{
  /*
   * This function calls the _f_dow function through the wrapper.
   */

  run1_1_Function(_f_dow, "dow");
}


void f_dowstr(void)
{
  /*
   * This function calls the _f_dowstr function through the wrapper.
   */

  run1_1_Function(_f_dowstr, "dowstr");
}


void f_tstr(void)
{
  /*
   * This function calls the _f_tstr function through the wrapper.
   */

  run2_1_Function(_f_tstr, "tstr");
}


Object _f_TOhms(Object n, int *err)
{
  /*
   * This function converts from decimal format to HH.MMSS format.
   *
   * ->HMS(x) = x - 0.4*(x - floor(x)) - 0.004*(60*x - floor(60*x))
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = __TOhms(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_hmsTO(Object n, int *err)
{
  /*
   * This function converts from HH.MMSS format to decimal format.
   *
   *                x - floor(x)   100*x - floor(100*x)
   * HMS->(x) = x + ------------ + --------------------
   *                     1.5                90
   */

  if (type(n) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = __hmsTO(n.value.real);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_dateAdd(Object n, Object p, int *err)
{
  /*
   * This function adds a number of days to a date, in MM.DDYYYY format.
   * The number of days is simply added to the tm_mday field of struct tm.
   * The normalization functions of mktime do the rest.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    int m, d, y;
    time_t timeT;
    struct tm tms;

    *err = ERR_NOERR;
    __extractMDY(n.value.real, &m, &d, &y);

    tms.tm_mon = m - 1;
    tms.tm_mday = d + p.value.real;
    tms.tm_year = y - 1900;
    tms.tm_hour = tms.tm_min = tms.tm_sec = tms.tm_isdst = 0;

    timeT = mktime(&tms);

    if (timeT == -1) {
      *err = ERR_INVALIDDATE;
      return p;
    }

    n.value.real = __createMDY(tms.tm_mon + 1,
			       tms.tm_mday,
			       tms.tm_year + 1900);
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_ddays(Object n, Object p, int *err)
{
  /*
   * This function returns the number of days between two dates.
   * Two time_t values are calculated and subtracted. The result is the
   * number of seconds between the two dates. Since there are 60 * 60 * 24
   * seconds in a day, we divide the number of seconds by that value to
   * get the number of days.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    int m1, d1, y1, m2, d2, y2;
    time_t time1, time2;
    struct tm tm1, tm2;

    *err = ERR_NOERR;

    __extractMDY(n.value.real, &m1, &d1, &y1);
    __extractMDY(p.value.real, &m2, &d2, &y2);

    tm1.tm_mon = m1 - 1;
    tm1.tm_mday = d1;
    tm1.tm_year = y1 - 1900;
    tm1.tm_hour = tm1.tm_min = tm1.tm_sec = tm1.tm_isdst = 0;

    tm2.tm_mon = m2 - 1;
    tm2.tm_mday = d2;
    tm2.tm_year = y2 - 1900;
    tm2.tm_hour = tm2.tm_min = tm2.tm_sec = tm2.tm_isdst = 0;

    time1 = mktime(&tm1);
    time2 = mktime(&tm2);

    if (time1 == -1 || time2 == -1) {
      *err = ERR_INVALIDDATE;
      return n;
    }

    n.value.real = difftime(time2, time1);
    n.value.real /= 60 * 60 * 24;
    n.value.real = (int)(n.value.real + copysign(.5, n.value.real));
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_hmsAdd(Object n, Object p, int *err)
{
  /*
   * This function adds two times in HH.MMSS format. To do that, it simply
   * converts the times into decimal notation, adds them, and converts them
   * back into HH.MMSS format.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = __TOhms(__hmsTO(n.value.real) + __hmsTO(p.value.real));
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_hmsSub(Object n, Object p, int *err)
{
  /*
   * This function subtracts two times in HH.MMSS format. To do that,
   * it simply converts the times into decimal notation, subtracts
   * them, and converts them back into HH.MMSS format.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    *err = ERR_NOERR;
    n.value.real = __TOhms(__hmsTO(n.value.real) - __hmsTO(p.value.real));
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_dow(Object n, int *err)
{
  /*
   * This function returns the day of the week, of a date given in
   * MM.DDYYYY format.
   * Returns 0 for Sunday, 1 for Monday, etc.
   */
  
  if (type(n) == TYPE_REAL) {
    int m, d, y;

    __extractMDY(n.value.real, &m, &d, &y);

    n.value.real = __dow(m, d, y);
    *err = n.value.real == -1 ? ERR_INVALIDDATE : ERR_NOERR;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_dowstr(Object n, int *err)
{
  /*
   * Returns the three-letter abbrev. of the week day of the given date
   * in MM.DDYYYY format.
   */

  if (type(n) == TYPE_REAL) {
    int m, d, y, wd;

    *err = ERR_NOERR;
    
    __extractMDY(n.value.real, &m, &d, &y);
    wd = __dow(m, d, y);

    if (wd == -1) {
      *err = ERR_INVALIDDATE;
      return n;
    }

    n.type = TYPE_STR;
    n.value.str = strdup(wdays[wd]);
    if (!n.value.str)
      *err = ERR_NOTENOUGHMEMORY;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


Object _f_tstr(Object n, Object p, int *err)
{
  /*
   * Returns a string representing the date and time, in the format
   * "WWW MM/DD/YYYY  HH:MM:SS"
   * (WWW is the three-letter weekday)
   *
   * NOTE: The HP48 version of this command uses just two digits for the
   * year, but I changed to four to be Y2K compliant.
   */

  if (type(n) == TYPE_REAL && type(p) == TYPE_REAL) {
    int m, d, y, h, min, s, dow;
    char *str = (char *) malloc(25);

    *err = ERR_NOERR;

    __extractMDY(n.value.real, &m, &d, &y);
    dow = __dow(m, d, y);

    if (dow == -1) {
      *err = ERR_INVALIDDATE;
      return n;
    }

    /* Extract hours, minutes and seconds */
    h = p.value.real;
    p.value.real -= h;
    p.value.real *= 1e2;
    min = p.value.real;
    p.value.real -= min;
    p.value.real *= 1e2;
    p.value.real += .5;
    s = p.value.real;

    sprintf(str, "%s %02d/%02d/%04d  %02d:%02d:%02d",
	    wdays[dow], m, d, y, h, min, s);

    n.type = TYPE_STR;
    n.value.str = str;
  } else
    *err = ERR_BADARGUMENTTYPE;

  return n;
}


void __extractMDY(double mdy, int *m, int *d, int *y)
{
  /*
   * Extracts the month, day and year from a date in MM.DDYYYY format.
   */

  *m = mdy;

  mdy -= *m;
  mdy *= 1e2;
  *d = mdy;

  mdy -= *d;
  mdy *= 1e4;
  mdy += .5;
  *y = mdy;
}


int __dow(int m, int d, int y)
{
  /*
   * Returns the week day: 0 for Sunday, 1 for Monday, etc. or -1 if
   * the date is invalid.
   */
  time_t timeT;
  struct tm tms;

  tms.tm_mon = m - 1;
  tms.tm_mday = d;
  tms.tm_year = y - 1900;
  tms.tm_hour = tms.tm_min = tms.tm_sec = tms.tm_isdst = 0;

  timeT = mktime(&tms);

  return (timeT == -1) ? -1 : tms.tm_wday;
}
