/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.service.utils;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.util.Preconditions;
import org.apache.hadoop.util.VersionInfo;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.LocalResourceType;
import org.apache.hadoop.yarn.api.records.LocalResourceVisibility;
import org.apache.hadoop.yarn.api.records.URL;
import org.apache.hadoop.yarn.service.exceptions.BadClusterStateException;
import org.apache.hadoop.yarn.service.exceptions.SliderException;
import org.apache.hadoop.yarn.service.utils.ServiceUtils;
import org.apache.hadoop.yarn.util.Records;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CoreFileSystem {
    private static final Logger log = LoggerFactory.getLogger(CoreFileSystem.class);
    private static final String UTF_8 = "UTF-8";
    protected final FileSystem fileSystem;
    protected final Configuration configuration;

    public CoreFileSystem(FileSystem fileSystem, Configuration configuration) {
        Preconditions.checkNotNull((Object)fileSystem, (Object)"Cannot create a CoreFileSystem with a null FileSystem");
        Preconditions.checkNotNull((Object)configuration, (Object)"Cannot create a CoreFileSystem with a null Configuration");
        this.fileSystem = fileSystem;
        this.configuration = configuration;
    }

    public CoreFileSystem(Configuration configuration) throws IOException {
        Preconditions.checkNotNull((Object)configuration, (Object)"Cannot create a CoreFileSystem with a null Configuration");
        this.fileSystem = FileSystem.get((Configuration)configuration);
        this.configuration = configuration;
    }

    public Path getTempPathForCluster(String clustername) {
        Path clusterDir = this.buildClusterDirPath(clustername);
        return new Path(clusterDir, "tmp");
    }

    public FileSystem getFileSystem() {
        return this.fileSystem;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("CoreFileSystem{");
        sb.append("fileSystem=").append(this.fileSystem.getUri()).append('}');
        return sb.toString();
    }

    public Path buildClusterDirPath(String clustername) {
        Preconditions.checkNotNull((Object)clustername);
        Path path = this.getBaseApplicationPath();
        return new Path(path, "services/" + clustername);
    }

    public Path buildClusterUpgradeDirPath(String clusterName, String version) {
        Preconditions.checkNotNull((Object)clusterName);
        Preconditions.checkNotNull((Object)version);
        return new Path(this.buildClusterDirPath(clusterName), "upgrade/" + version);
    }

    public void deleteClusterUpgradeDir(String clusterName, String version) throws IOException {
        Preconditions.checkNotNull((Object)clusterName);
        Preconditions.checkNotNull((Object)version);
        Path upgradeCluster = this.buildClusterUpgradeDirPath(clusterName, version);
        this.fileSystem.delete(upgradeCluster, true);
    }

    public Path buildKeytabInstallationDirPath(String keytabFolder) {
        Preconditions.checkNotNull((Object)keytabFolder);
        Path path = this.getBaseApplicationPath();
        return new Path(path, "keytabs/" + keytabFolder);
    }

    public Path buildKeytabPath(String keytabDir, String keytabName, String clusterName) {
        Path homePath = this.getHomeDirectory();
        Path baseKeytabDir = keytabDir != null ? new Path(homePath, keytabDir) : new Path(this.buildClusterDirPath(clusterName), "keytabs");
        return keytabName == null ? baseKeytabDir : new Path(baseKeytabDir, keytabName);
    }

    public Path buildResourcePath(String resourceFolder) {
        Preconditions.checkNotNull((Object)resourceFolder);
        Path path = this.getBaseApplicationPath();
        return new Path(path, "resources/" + resourceFolder);
    }

    public Path buildResourcePath(String dirName, String fileName) {
        Preconditions.checkNotNull((Object)dirName);
        Preconditions.checkNotNull((Object)fileName);
        Path path = this.getBaseApplicationPath();
        return new Path(path, "resources/" + dirName + "/" + fileName);
    }

    public void createWithPermissions(Path dir, FsPermission clusterPerms) throws IOException, BadClusterStateException {
        if (this.fileSystem.isFile(dir)) {
            throw new BadClusterStateException("Cannot create a directory over a file %s", dir);
        }
        log.debug("mkdir {} with perms {}", (Object)dir, (Object)clusterPerms);
        this.fileSystem.getConf().set("fs.permissions.umask-mode", "000");
        this.fileSystem.mkdirs(dir, clusterPerms);
        this.fileSystem.setPermission(dir, clusterPerms);
    }

    public void verifyClusterDirectoryNonexistent(String clustername, Path clusterDirectory) throws IOException, SliderException {
        if (this.fileSystem.exists(clusterDirectory)) {
            throw new SliderException(75, "Service Instance \"%s\" already exists and is defined in %s", clustername, clusterDirectory);
        }
    }

    public void verifyDirectoryNonexistent(Path clusterDirectory) throws IOException, SliderException {
        if (this.fileSystem.exists(clusterDirectory)) {
            log.error("Dir {} exists: {}", (Object)clusterDirectory, (Object)this.listFSDir(clusterDirectory));
            throw new SliderException(75, "Service Instance dir already exists: %s", clusterDirectory);
        }
    }

    public void verifyDirectoryWriteAccess(Path dirPath) throws IOException, SliderException {
        this.verifyPathExists(dirPath);
        Path tempFile = new Path(dirPath, "tmp-file-for-checks");
        try {
            FSDataOutputStream out = this.fileSystem.create(tempFile, true);
            IOUtils.closeStream((Closeable)out);
            this.fileSystem.delete(tempFile, false);
        }
        catch (IOException e) {
            log.warn("Failed to create file {}: {}", (Object)tempFile, (Object)e);
            throw new BadClusterStateException(e, "Unable to write to directory %s : %s", dirPath, e.toString());
        }
    }

    public void verifyPathExists(Path path) throws IOException {
        if (!this.fileSystem.exists(path)) {
            throw new FileNotFoundException(path.toString());
        }
    }

    public void verifyFileExists(Path path) throws IOException {
        FileStatus status = this.fileSystem.getFileStatus(path);
        if (!status.isFile()) {
            throw new FileNotFoundException("Not a file: " + path.toString());
        }
    }

    public boolean isFile(Path path) {
        if (path == null) {
            return false;
        }
        boolean isFile = false;
        try {
            FileStatus status = this.fileSystem.getFileStatus(path);
            if (status.isFile()) {
                isFile = true;
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
        return isFile;
    }

    public Path getBaseApplicationPath() {
        String configuredBasePath = this.configuration.get("yarn.service.base.path", this.getHomeDirectory() + "/" + ".yarn");
        return new Path(configuredBasePath);
    }

    public Path getDependencyTarGzip() {
        Path dependencyLibTarGzip = null;
        String configuredDependencyTarballPath = this.configuration.get("yarn.service.framework.path");
        if (configuredDependencyTarballPath != null) {
            dependencyLibTarGzip = new Path(configuredDependencyTarballPath);
        }
        if (dependencyLibTarGzip == null) {
            dependencyLibTarGzip = new Path(String.format("/yarn-services/%s/", VersionInfo.getVersion()), "service-dep.tar.gz");
        }
        return dependencyLibTarGzip;
    }

    public Path getHomeDirectory() {
        return this.fileSystem.getHomeDirectory();
    }

    public LocalResource createAmResource(Path destPath, LocalResourceType resourceType, LocalResourceVisibility visibility) throws IOException {
        FileStatus destStatus = this.fileSystem.getFileStatus(destPath);
        LocalResource amResource = (LocalResource)Records.newRecord(LocalResource.class);
        amResource.setType(resourceType);
        if (visibility == null) {
            visibility = LocalResourceVisibility.APPLICATION;
        }
        amResource.setVisibility(visibility);
        amResource.setResource(URL.fromPath((Path)this.fileSystem.resolvePath(destStatus.getPath())));
        amResource.setTimestamp(destStatus.getModificationTime());
        amResource.setSize(destStatus.getLen());
        return amResource;
    }

    public Map<String, LocalResource> submitDirectory(Path srcDir, String destRelativeDir) throws IOException {
        FileStatus[] fileset = this.fileSystem.listStatus(srcDir);
        HashMap<String, LocalResource> localResources = new HashMap<String, LocalResource>(fileset.length);
        for (FileStatus entry : fileset) {
            LocalResource resource = this.createAmResource(entry.getPath(), LocalResourceType.FILE, LocalResourceVisibility.APPLICATION);
            String relativePath = destRelativeDir + "/" + entry.getPath().getName();
            localResources.put(relativePath, resource);
        }
        return localResources;
    }

    public LocalResource submitJarWithClass(Class clazz, Path tempPath, String subdir, String jarName) throws IOException, SliderException {
        File localFile = ServiceUtils.findContainingJarOrFail(clazz);
        return this.submitFile(localFile, tempPath, subdir, jarName);
    }

    public LocalResource submitFile(File localFile, Path tempPath, String subdir, String destFileName) throws IOException {
        Path src = new Path(localFile.toString());
        Path subdirPath = new Path(tempPath, subdir);
        this.fileSystem.mkdirs(subdirPath);
        Path destPath = new Path(subdirPath, destFileName);
        log.debug("Copying {} (size={} bytes) to {}", new Object[]{localFile, localFile.length(), destPath});
        this.fileSystem.copyFromLocalFile(false, true, src, destPath);
        return this.createAmResource(destPath, LocalResourceType.FILE, LocalResourceVisibility.APPLICATION);
    }

    public void submitTarGzipAndUpdate(Map<String, LocalResource> providerResources) throws IOException, BadClusterStateException {
        Path dependencyLibTarGzip = this.getDependencyTarGzip();
        LocalResource lc = this.createAmResource(dependencyLibTarGzip, LocalResourceType.ARCHIVE, LocalResourceVisibility.APPLICATION);
        providerResources.put("service_dep", lc);
    }

    public void copyLocalFileToHdfs(File localPath, Path destPath, FsPermission fp) throws IOException {
        if (localPath == null || destPath == null) {
            throw new IOException("Either localPath or destPath is null");
        }
        this.fileSystem.getConf().set("fs.permissions.umask-mode", "000");
        this.fileSystem.mkdirs(destPath.getParent(), fp);
        log.info("Copying file {} to {}", (Object)localPath.toURI(), (Object)destPath);
        this.fileSystem.copyFromLocalFile(false, true, new Path(localPath.getPath()), destPath);
        this.fileSystem.setPermission(destPath, fp);
    }

    public void copyHdfsFileToLocal(Path hdfsPath, File destFile) throws IOException {
        if (hdfsPath == null || destFile == null) {
            throw new IOException("Either hdfsPath or destPath is null");
        }
        log.info("Copying file {} to {}", (Object)hdfsPath.toUri(), (Object)destFile.toURI());
        Path destPath = new Path(destFile.getPath());
        this.fileSystem.copyToLocalFile(hdfsPath, destPath);
    }

    public String listFSDir(Path path) throws IOException {
        FileStatus[] stats = this.fileSystem.listStatus(path);
        StringBuilder builder = new StringBuilder();
        for (FileStatus stat : stats) {
            builder.append(stat.getPath().toString()).append("\t").append(stat.getLen()).append("\n");
        }
        return builder.toString();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public String cat(Path path) throws IOException {
        String string;
        FileStatus status = this.fileSystem.getFileStatus(path);
        byte[] b = new byte[(int)status.getLen()];
        FSDataInputStream in = null;
        try {
            in = this.fileSystem.open(path);
            int count = in.read(b);
            string = new String(b, 0, count, UTF_8);
        }
        catch (Throwable throwable) {
            IOUtils.closeStream(in);
            throw throwable;
        }
        IOUtils.closeStream((Closeable)in);
        return string;
    }
}

