#!/usr/bin/perl
# $Id: Generic.pm,v 1.6 2003/12/10 03:52:30 nothingmuch Exp $

package Object::Meta::Plugin::Useful::Generic; # an extended base class with some logical features $$$ ## rename to Usefull::Generic;

use strict;
use warnings;

use base 'Object::Meta::Plugin::Useful';

our $VERSION = 0.01;

sub new {
	my $pkg = shift;
	
	bless {
		exports => [],
	}, $pkg;
}

sub export { # utility method: export a list of method names
	my $self = shift;
	my @try_export = @_;
	
	my %tested = map { $_, undef } @{ $self->{exports} };
	
	push @{ $self->{exports} }, grep {
		(not exists $tested{$_}) and $tested{$_} = 1 # make sure we didn't pass this one already
		and $self->can($_) or (warnings::warnif($self,"Export of undefined method $_ attempted") and undef); # make sure we can use the method. UNIVERSAL::can should be replaced if magic is going on. To shut it up, 'no warnings Plugin::Class'.
	} @try_export;
	
	@{ $self->{exports} }; # number on scalar, list on list
}

sub exports {
	my $self = shift;
	@{ $self->{exports} };
}

1; # Keep your mother happy.

__END__

=pod

=head1 NAME

Object::Meta::Plugin::Useful::Generic - A generic useful plugin base class.

=head1 SYNOPSIS

	package MyFoo;
	use base "Object::Meta::Plugin::Useful::Generic";

	sub new {
		my $pkg = shift;
		my $self = $pkg->SUPER::new(@_);
		$self->export(qw/foo bar/);
	}

	sub foo {
		# ...
	}

	sub bar {
		# ...
	}

=head1 DESCRIPTION

This provides a very simple base class for a plugin. It uses the method C<export> to explicitly mark a method name for exporting. When L<Object::Meta::Plugin::Useful>'s C<init> hits

=head1 METHODS

=over 4

=item exports

Returns the list of method names as created by C<export>.

=item export METHODS ...

This method takes a list of method names, and makes sure they are all implemented (C<$self->can($method)>) and so forth. It then makes notes of what remains, and will return these values when the exports method is called by the standard export list implementation.

=back

=head1 CAVEATS

=over 4

=item *

Will emit warnings if lexical warnings are asked for. It will bark when C<$self->can($method)> is not happy. You can suppress it with

	no warnings 'MyFoo';
	
Or whatever.

This also means that you will need to

	use warnings::register;


The errors are not fatal like they are in L<Object::Meta::Plugin::Host>, because this plugin implementation does not need to be necessarily plugged into the L<Object::Meta::Plugin::Host> implementation.

=back

=head1 BUGS

Nothing I know of.

=head1 TODO

Nothing right now.

=head1 ACKNOWLEDGMENTS

=over 4

=item Mike Castle, for spotting a typo.

=back

=head1 COPYRIGHT & LICENSE

	Copyright 2003 Yuval Kogman. All rights reserved.
	This program is free software; you can redistribute it
	and/or modify it under the same terms as Perl itself.

=head1 AUTHOR

Yuval Kogman <nothingmuch@woobling.org>

=head1 SEE ALSO

L<Object::Meta::Plugin>, L<Object::Meta::Plugin::Useful>, L<Object::Meta::Plugin::Useful::Meta>, L<Object::Meta::Plugin::Useful::Greedy>.

=cut
