/*
 * httpd.c: httpd server mode.
 *
 * Copyright (c) 2009-2010, NIPPON TELEGRAPH AND TELEPHONE CORPORATION
 */

#include "pg_reporter.h"
#include "pgut/pgut-httpd.h"

#include <dirent.h>
#include <sys/stat.h>
#include <time.h>

#include <libxslt/transform.h>

static void cgi_error(const char *message);
static const char *httpd_callback(StringInfo out, const http_request *request);
static const char *makeDatabaseReport(StringInfo out, List *vars, const char *xml);
static xmlDocPtr makeFileList(const char *dirPath);
static xmlDocPtr makeDatabaseList(void);
static char *decode(const char *str, int length);


/*-------------------------------------------------------------------------
 * cgi-mode
 *-------------------------------------------------------------------------
 */

int
cgi_main(const char *dir)
{
	char		   *query_string;
	StringInfoData 	body;			/* body text of the report */
	const char	   *result;			/* content-type or error code */
	const char	   *xml;			/* template name */
	List		   *vars;

	if (chdir(dir) < 0)
		elog(ERROR, "could not change directory to \"%s\"", dir);

	pgut_abort_level = FATAL;

	if ((query_string = getenv("QUERY_STRING")) == NULL)
	{
		cgi_error("QUERY_STRING is not defined");
		return 1;
	}

	/* parse QUERY_STRING and make a variable table */
	vars = makeVariables(query_string);
	if (vars == NULL)
	{
		cgi_error("invalid URL request");
		return 1;
	}

	/* retrieve template name from URL parameter */
	if ((xml = getVariable(vars, "template")) == NULL)
	{
		elog(WARNING, "\"template\" option was not found in url parameter");
		result = "*";
	}
	else
	{
		/* Make a database report. */
		initStringInfo(&body);
		result = makeDatabaseReport(&body, vars, xml);
	}

	/* content-type starts with '*' means an error. */
	if (result[0] == '*')
	{
		cgi_error(result);
		return 1;
	}

	printf("Content-Type: %s\r\n", result);
	fwrite(body.data, 1, body.len, stdout);

	list_destroy(vars, freeVariable);
	termStringInfo(&body);
	return 0;
}

/* write error messages for web browsers */
static void
cgi_error(const char *message)
{
	printf("Content-Type: text/html\r\n");
	printf("\n");
	printf("<html>\n");
	printf("<head>\n");
	printf("<title>Error</title>\n");
	printf("</head>\n");
	printf("<body>\n");
	printf("%s error:\n", PROGRAM_NAME);
	printf("<br />\n");
	printf("%s<br />\n", message);
	printf("</body>\n");
	printf("</html>\n");
}

/*-------------------------------------------------------------------------
 * httpd-mode
 *-------------------------------------------------------------------------
 */

int
httpd_main(const char *dir, int port, const char *listen_addresses)
{
	if (chdir(dir) < 0)
		elog(ERROR, "could not change directory to \"%s\"", dir);
	if (listen_addresses == NULL)
		listen_addresses = "localhost";

	elog(INFO, "httpd: start (port=%d, dir=%s)", port, dir);
	pgut_abort_level = FATAL;
	pgut_httpd(port, listen_addresses, httpd_callback);

	return 0;
}

static const char *
httpd_callback(StringInfo out, const http_request *request)
{
	const char *path;
	char	   *pos;
	char	   *id = NULL;
	const char *xml;
	List	   *vars = NIL;
	const char *result = "*500";

	if (request->url[0] != '/')
	{
		elog(WARNING, "URL parameter is invalid");
		return "*400";
	}

	path = request->url;
	while (*path == '/') { path++; }
	id = pgut_strdup(path);

	pos = strchr(id, '/');
	if (pos == NULL)
	{
		xml = XML_DEFAULT;
	}
	else
	{
		size_t		i;

		pos[0] = '\0';
		pos++;

		/* remove trailing '/' */
		for (i = strlen(pos); i > 0 && pos[i - 1] == '/'; i--)
			pos[i - 1] = '\0';

		if (pos[0] == '\0')
			xml = XML_DEFAULT;
		else
			xml = pos;
	}

	elog(DEBUG2, "request: id=%s, xml=%s, params=%s",
		id, xml, (request->params ? request->params : "(null)"));

	/* return a database list if root directory is required. */
	if (id[0] == '\0')
	{
		xmlDocPtr	xml = makeDatabaseList();
		if (xml != NULL)
		{
			result = applyStylesheet(out, xml, XSL_DBNAMES, NIL);
			xmlFreeDoc(xml);
		}
		else
		{
			result = "*404";
			goto done;
		}
		goto done;
	}

	/* return a file list if id = FILES_DIR */
	if (strcmp(id, FILES_DIR) == 0)
	{
		xmlDocPtr	xml = makeFileList(path);

		if (xml != NULL)
		{
			result = applyStylesheet(out, xml, XSL_DIR, NIL);
			xmlFreeDoc(xml);
		}
		else
		{
			result = "*404";
		}
		goto done;
	}

	/* make a variable table from URL parameters, id and template */
	vars = makeVariables(request->params);
	vars = addVariable(vars, "id", id);
	vars = addVariable(vars, "template", xml);

	/* Make a database report. */
	result = makeDatabaseReport(out, vars, xml);

done:
	free(id);
	list_destroy(vars, freeVariable);
	xsltCleanupGlobals();
	xmlCleanupParser();
	return result;
}

/*-------------------------------------------------------------------------
 * common routines for httpd and cgi modes
 *-------------------------------------------------------------------------
 */

static const char *
makeDatabaseReport(StringInfo out, List *vars, const char *xml)
{
	const char	   *result;
	struct stat		st;
	char			path[MAXPGPATH];
	const char	   *id;
	Database	   *db;

	/* Make a report if a XML file is found. */
	snprintf(path, sizeof(path), "template/%s.xml", xml);
	if (stat(&path[0], &st) != 0)
	{
		elog(WARNING, "template not found: %s", path);
		return "*404";
	}

	/* Find a database connection information by "id" parameter. */
	if ((id = getVariable(vars, "id")) == NULL)
		return "*500";
	if ((db = getDatabase(id)) == NULL)
		return "*404";

	/* Setup connection information. */
	host = db->host;
	port = db->port;
	dbname = db->database;
	username = db->username;
	password = db->password;
	prompt_password = NO;

	/* Make a report. */
	vars = addVariable(vars, "host", host ? host : "");
	vars = addVariable(vars, "port", port ? port : "");
	vars = addVariable(vars, "database", dbname ? dbname : "");
	result = makeReport(out, path, vars);

	/* Clear connection information.*/
	host = NULL;
	port = NULL;
	dbname = NULL;
	username = NULL;
	password = NULL;

	freeDatabase(db);

	return result;
}

/*
 * <dir>
 *   <file>
 *     <name></name>
 *     <size></size>
 *     <mtime></mtime>
 *   </file>
 *   ...
 * </dir>
 */
static xmlDocPtr
makeFileList(const char *dirUrl)
{
	xmlDocPtr		doc;
	xmlNodePtr		root;
	DIR			   *dir;
	struct dirent  *dp;
	char			dirPath[MAXPGPATH];
	size_t			len;

	strlcpy(dirPath, dirUrl, MAXPGPATH);
	len = strlen(dirPath);
	while (len > 1 && dirPath[len - 1] == '/')
		dirPath[--len] = '\0';

	if (dirPath[0] == '\0' ||
		dirPath[0] == '/' ||
		(dir = opendir(dirPath)) == NULL)
	{
		elog(WARNING, "cannot open directory \"%s\": %s", dirUrl, strerror(errno));
		return NULL;
	}

	/* make a document */
	doc = xmlNewDoc((const xmlChar *) "1.0");

	/* <dir path="..."> */
	root = xmlNewDocNode(doc, NULL, (const xmlChar *) "dir", NULL);
	xmlDocSetRootElement(doc, root);
	xmlNewProp(root, (const xmlChar *) "path", (const xmlChar *) dirPath);

	while ((dp = readdir(dir)) != NULL)
	{
		xmlNodePtr		file;
		struct stat		st;
		char			mtime[64];
		struct tm	   *tm;
		char			path[MAXPGPATH];

		if (dp->d_name[0] == '.')
			continue;	/* hide dot files */

		snprintf(path, sizeof(path), "%s/%s", dirPath, dp->d_name);
		if (stat(path, &st) != 0)
			continue;

		file = xmlNewChild(root, NULL, (const xmlChar *) "file", NULL);

		/* <name> */
		xmlNewTextChild(file, NULL,
			(const xmlChar *) "name",
			(const xmlChar *) dp->d_name);

		/* <size> */
		if (!S_ISDIR(st.st_mode))
		{
			char	size[32];

			/* round up to KB */
			snprintf(size, sizeof(size), "%lu",
				(unsigned long) ((st.st_size + 1023) / 1024));
			xmlNewTextChild(file, NULL,
				(const xmlChar *) "size",
				(const xmlChar *) size);
		}

		/* <mtime> */
		tm = localtime(&st.st_mtime);
		strftime(mtime, sizeof(mtime), "%c", tm);
		xmlNewTextChild(file, NULL,
			(const xmlChar *) "mtime",
			(const xmlChar *) mtime);
	}

	closedir(dir);

	return doc;
}

/*
 * <dbnames>
 *   <dbname>
 *     <id>id</id>
 *     <host>host</host>
 *     <port>port</port>
 *     <database>database</database>
 *   </dbname>
 *   ...
 * </dbnames>
 */
static xmlDocPtr
makeDatabaseList(void)
{
	xmlDocPtr		doc;
	xmlNodePtr		root;
	ListCell	   *cell;
	List		   *dbs;

	dbs = getDatabases();

	/* make a XML document */
	doc = xmlNewDoc((const xmlChar *) "1.0");

	/* make a root node named "dbnames" */
	root = xmlNewDocNode(doc, NULL, (const xmlChar *) "dbnames", NULL);
	xmlDocSetRootElement(doc, root);

	/* add databases */
	foreach(cell, dbs)
	{
		Database   *db = lfirst(cell);
		xmlNodePtr	dbname;

		dbname = xmlNewChild(root, NULL, (const xmlChar *) "dbname", NULL);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "id",
			(const xmlChar *) db->id);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "host",
			(const xmlChar *) db->host);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "port",
			(const xmlChar *) db->port);
		xmlNewTextChild(dbname, NULL,
			(const xmlChar *) "database",
			(const xmlChar *) db->database);
	}

	list_destroy(dbs, freeDatabase);

	return doc;
}

/* decode uuencoded string */
static char *
decode(const char *str, int length)
{
    int				i, pos;
	char		   *buf;
	unsigned char	c;
	char			wk[3];

	buf = pgut_malloc(length + 1);
	for (i = 0, pos = 0 ; i < length ; i++)
	{
		c = str[i];

		if (c == '%')
		{
			i++;
			memcpy(wk, str+i, 2);
			wk[2] = 0x00;

			c = (char) strtol(wk, NULL, 16);
			i++;
		}

		buf[pos++] = c;
	}

	buf[pos] = 0x00;

	return buf;
}

/*-------------------------------------------------------------------------
 * variable managements
 *-------------------------------------------------------------------------
 */

/* make a variable table */
List *
makeVariables(const char *parameters)
{
	List	   *vars = NIL;
	const char *name;

	/* fast path for no vars */
	if (parameters == NULL || parameters[0] == '\0')
		return NIL;

	name = parameters;
	for (;;)
	{
		Variable	   *var;
		const char	   *value;

		/* variable format is 'name=value' */
		if ((value = strchr(name, '=')) == NULL || name == value)
		{
			elog(WARNING, "bad url format \"%s\"", parameters);
			break;
		}

		var = pgut_new(Variable);
		var->name = decode(name, value - name);
		value++;

		vars = lappend(vars, var);

		/* split with '&' */
		name = strchr(value, '&');
		if (name == NULL)
			name = value + strlen(value);
		if (name == value)
		{
			/* format is "name=&..." */
			var->value = pgut_strdup("");
		}
		else
		{
			/* format is "name=value&..." */
			var->value = decode(value, name - value);
		}

		/* goto next var or exit */
		if (*name == '&')
			name++;
		else
			break;
	}

	return vars;
}

/* add a variable to the variable table */
List *
addVariable(List *vars, const char *name, const char *value)
{
	Variable *var;

	var = pgut_new(Variable);
	var->name = pgut_strdup(name);
	var->value = pgut_strdup(value);

	/* XXX: unique checks needed? */
	return lappend(vars, var);
}

/* find a variable from variable table */
const char *
getVariable(List *vars, const char *name)
{
	ListCell *cell;

	foreach(cell, vars)
	{
		Variable *var = lfirst(cell);
		if (strcmp(var->name, name) == 0)
			return var->value;
	}

	return NULL;
}

/* free Variable object */
void
freeVariable(Variable *var)
{
	if (var == NULL)
		return;

	free(var->name);
	free(var->value);
	free(var);
}
