/*--------------------------------------------------------------------
 * FILE:
 *     ssl.c
 *
 * NOTE:
 *     This file is composed of the SSL functions.
 *
 *--------------------------------------------------------------------
 */
#include "postgres.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <ctype.h>
#include <time.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <arpa/inet.h>

#ifdef MULTIBYTE
#include "mb/pg_wchar.h"
#endif

#include "pgc_admin.h"
#define	MSG_BUF_SIZE	(256)

struct CRYPTO_dynlock_value
{
	pthread_mutex_t mutex;
};

DH *dh512 = NULL;
DH *dh1024 = NULL;
SSL_Server_Info * DH_Server_Info = NULL;

/* This array will store all of the mutexes available to OpenSSL. */
static pthread_mutex_t *mutex_buf = NULL;

void PGC_SSL_Init(void);
int PGC_SSL_Verify_Callback(int ok, X509_STORE_CTX *store);
long PGC_SSL_Post_Connection_Check(SSL_Info *ssl_tbl, char *host);
void PGC_SSL_Seed_Prng(void);
int PGC_SSL_Send(SSL_Info *ssl_tbl, char * buf, int length );
int PGC_SSL_Read(SSL_Info *ssl_tbl, char * buf, int length );
int PGC_SSL_Thread_Setup(void);
int PGC_SSL_Thread_Cleanup(void);
void PGC_SSL_Server_Init_DH(void);
DH * PGC_SSL_Server_DH_Callback(SSL *ssl, int is_export, int keylength);
SSL_CTX * PGC_SSL_Server_Setup_Ctx(SSL_Server_Info * ssl_server);
SSL_CTX * PGC_SSL_Client_Setup_Ctx(SSL_Server_Info * ssl_server);
void PGC_Close_SSL(SSL_Info * ssl_tbl);
void PGC_Clear_SSL(SSL_Info * ssl_tbl,int flag);
SSL_Info * PGC_Create_Admin_Send_SSL(SSL_Info * ssl_tbl, char * hostName,  uint16_t portNumber);
SSL_Info * PGC_Create_Probe_Send_SSL(SSL_Info * ssl_tbl, char * hostName,  uint16_t portNumber);

static void locking_function(int mode, int n, const char * file, int line);
static struct CRYPTO_dynlock_value * dyn_create_function(const char *file, int line);
static void dyn_lock_function(int mode, struct CRYPTO_dynlock_value *l, const char *file, int line);
static void dyn_destroy_function(struct CRYPTO_dynlock_value *l, const char *file, int line);
static unsigned long id_function(void);
static SSL_Info * create_send_ssl(SSL_Info * ssl_tbl, char * hostName,  uint16_t portNumber, SSL_Server_Info * ssl_server);


void 
PGC_SSL_Init(void)
{
	char * func = "PGC_SSL_Init()";
    if (!PGC_SSL_Thread_Setup() || !SSL_library_init())
    {
		show_error("%s: PGC_SSL_Thread_Setup() or SSL_library_init() failed.",func);
        exit(-1);
    }
    SSL_load_error_strings();
}

int 
PGC_SSL_Verify_Callback(int ok, X509_STORE_CTX *store)
{
    char data[MSG_BUF_SIZE];
 
    if (!ok)
    {
        X509 *cert = X509_STORE_CTX_get_current_cert(store);
        int  depth = X509_STORE_CTX_get_error_depth(store);
        int  err = X509_STORE_CTX_get_error(store);
 
        X509_NAME_oneline(X509_get_issuer_name(cert), data, sizeof(data));
        X509_NAME_oneline(X509_get_subject_name(cert), data, sizeof(data));
        show_error("-Error with certificate at depth: %i\n", depth);
        show_error("  issuer   = %s\n", data);
        show_error("  subject  = %s\n", data);
        show_error("  err %i:%s\n", err, X509_verify_cert_error_string(err));
    }
 
    return ok;
}

long 
PGC_SSL_Post_Connection_Check(SSL_Info *ssl_tbl, char *host)
{
	char * func ="PGC_SSL_Post_Connection_Check()";
	X509 * cert = NULL;
    X509_NAME *subj = NULL;
	char      msg_data[MSG_BUF_SIZE]; 
    char      host_ip[IP_ADDR_LENGTH];
    int       extcount;
    int       ok = 0;
 
    if (!(cert = SSL_get_peer_certificate(ssl_tbl->ssl)) || !host)
	{
		show_error("%s: SSL_get_peer_certificate failed",func);
        goto err_occured;
	}
    if ((extcount = X509_get_ext_count(cert)) > 0)
    {
        int i;
 
        for (i = 0;  i < extcount;  i++)
        {
            char              *extstr;
            X509_EXTENSION    *ext;
 
            ext = X509_get_ext(cert, i);
            extstr = (char*) OBJ_nid2sn(OBJ_obj2nid(X509_EXTENSION_get_object(ext)));
            if (!strcmp(extstr, "subjectAltName"))
            {
                int                  j;
                unsigned char        *data;
                STACK_OF(CONF_VALUE) *val;
                CONF_VALUE           *nval;
                X509V3_EXT_METHOD    *meth;
                void                 *ext_str = NULL;
 
                if (!(meth = X509V3_EXT_get(ext)))
                    break;
                data = ext->value->data;

#if (OPENSSL_VERSION_NUMBER > 0x00907000L)
                if (meth->it)
                  ext_str = ASN1_item_d2i(NULL, &data, ext->value->length, ASN1_ITEM_ptr(meth->it));
                else
                  ext_str = meth->d2i(NULL, &data, ext->value->length);
#else
                ext_str = meth->d2i(NULL, &data, ext->value->length);
#endif
                val = meth->i2v(meth, ext_str, NULL);
                for (j = 0;  j < sk_CONF_VALUE_num(val);  j++)
                {
                    nval = sk_CONF_VALUE_value(val, j);
                    if (!strcmp(nval->name, "DNS") && !strcmp(nval->value, host))
                    {
                        ok = 1;
                        break;
                    }
                }
            }
            if (ok)
                break;
        }
    }
 
	memset(msg_data, 0, sizeof(msg_data));
	memset(host_ip, 0, sizeof(host_ip));
    if (!ok && (subj = X509_get_subject_name(cert)) &&
        X509_NAME_get_text_by_NID(subj, NID_commonName, msg_data, sizeof(msg_data)) > 0)
    {
        msg_data[sizeof(msg_data) -1] = 0;
		PGR_Set_Host_Name(host_ip, msg_data);
        if (strcasecmp(host_ip, host) != 0)
            goto err_occured;
    }
 
    X509_free(cert);
    return SSL_get_verify_result(ssl_tbl->ssl);
 
err_occured:
	show_error("%s: err_occured",func);
    if (cert)
        X509_free(cert);
    return X509_V_ERR_APPLICATION_VERIFICATION;
}

void 
PGC_SSL_Seed_Prng(void)
{
  RAND_load_file("/dev/urandom", 1024);
}

int 
PGC_SSL_Send(SSL_Info *ssl_tbl, char * buf, int length )
{
    int  send_len, total_send_len;
 
	for (total_send_len = 0;  total_send_len < length; )
	{
		send_len = SSL_write(ssl_tbl->ssl, buf + total_send_len, length - total_send_len);
		total_send_len += send_len;
		if (total_send_len == length)
		{
			return length;
		}
		else if (send_len <= 0)
		{
			return 0;
		}
	}
	return 0;
}

int
PGC_SSL_Read(SSL_Info *ssl_tbl, char * buf, int length )
{
	char * func = "PGC_SSL_Read()";
    int  read_len, total_read_len;
 
	for (total_read_len = 0;  total_read_len < length; )
	{
		read_len = SSL_read(ssl_tbl->ssl, buf + total_read_len, length - total_read_len);
		total_read_len += read_len;
		if (total_read_len == length)
		{
			return total_read_len;
		}
		else if (read_len < 0)
		{
			show_error("%s: read_len failed[%d]\n",func, read_len);
			break;
		}
	}
	return -1;
}

static void 
locking_function(int mode, int n, const char * file, int line)
{
	if (mode & CRYPTO_LOCK)
	{
		pthread_mutex_lock(&(mutex_buf[n]));
	}
	else
	{
		pthread_mutex_unlock(&(mutex_buf[n]));
	}
}

static struct CRYPTO_dynlock_value * 
dyn_create_function(const char *file, int line)
{
	struct CRYPTO_dynlock_value *value = NULL;
	value = (struct CRYPTO_dynlock_value *)malloc(sizeof(struct CRYPTO_dynlock_value));
	if (!value)
	{
		return NULL;
	}
	pthread_mutex_init(&(value->mutex), NULL);
	return value;
}

static void 
dyn_lock_function(int mode, struct CRYPTO_dynlock_value *l, const char *file, int line)
{
	if (mode & CRYPTO_LOCK)
	{
		pthread_mutex_lock(&(l->mutex));
	}
	else
	{
		pthread_mutex_unlock(&(l->mutex));
	}
}

static void 
dyn_destroy_function(struct CRYPTO_dynlock_value *l, const char *file, int line)
{
	pthread_mutex_destroy(&(l->mutex));
	free(l);
}

static unsigned long
id_function(void)
{
	return ((unsigned long) pthread_self());
}

int 
PGC_SSL_Thread_Setup(void)
{
	int i;
	mutex_buf = (pthread_mutex_t *)malloc(CRYPTO_num_locks( ) * sizeof(pthread_mutex_t));
	if (!mutex_buf)
	{
		return 0;
	}
	for (i = 0; i < CRYPTO_num_locks( ); i++)
	{
		pthread_mutex_init(&(mutex_buf[i]), NULL);
	}
	CRYPTO_set_id_callback(id_function);
	CRYPTO_set_locking_callback(locking_function);
	/* The following three CRYPTO_... functions are the OpenSSL functions
		for registering the callbacks we implemented above */
	CRYPTO_set_dynlock_create_callback(dyn_create_function);
	CRYPTO_set_dynlock_lock_callback(dyn_lock_function);
	CRYPTO_set_dynlock_destroy_callback(dyn_destroy_function);
	return 1;
}

int 
PGC_SSL_Thread_Cleanup(void)
{
	int i;

	if (!mutex_buf)
	{
		return 0;
	}
	CRYPTO_set_id_callback(NULL);
	CRYPTO_set_locking_callback(NULL);
	CRYPTO_set_dynlock_create_callback(NULL);
	CRYPTO_set_dynlock_lock_callback(NULL);
	CRYPTO_set_dynlock_destroy_callback(NULL);
	for (i = 0; i < CRYPTO_num_locks( ); i++)
	{
		pthread_mutex_destroy(&(mutex_buf[i]));
	}
	free(mutex_buf);
	mutex_buf = NULL;
	return 1;
}

void 
PGC_SSL_Server_Init_DH(void)
{
	char * func = "PGC_SSL_Server_Init_DH()";
    BIO *bio = NULL;
	char dh512_cert_file[PATH_MAX_LENGTH];
	char dh1024_cert_file[PATH_MAX_LENGTH];

	if (DH_Server_Info == NULL)
	{
		show_error("%s: DH_Server_Info is NULL",func);
		return;
	}
	memset(dh512_cert_file, 0, sizeof(dh512_cert_file));
	snprintf(dh512_cert_file, sizeof(dh512_cert_file),"%s/%s",DH_Server_Info->workPath, DH512_CERTFILE);

    bio = BIO_new_file(dh512_cert_file, "r");
    if (!bio)
	{
        show_error("%s: %s open failed",func, dh512_cert_file);
		return;
	}
    dh512 = PEM_read_bio_DHparams(bio, NULL, NULL, NULL);
    if (!dh512)
	{
        show_error("%s:DH parameters read failed from %s",func, dh512_cert_file);
		BIO_free(bio);
		return;
	}
    BIO_free(bio);

	memset(dh1024_cert_file, 0, sizeof(dh1024_cert_file));
	snprintf(dh1024_cert_file, sizeof(dh1024_cert_file),"%s/%s",DH_Server_Info->workPath, DH1024_CERTFILE);
    bio = BIO_new_file( dh1024_cert_file, "r");
    if (!bio)
	{
        show_error("%s: %s open failed",func, dh1024_cert_file);
		return;
	}
    dh1024 = PEM_read_bio_DHparams(bio, NULL, NULL, NULL);
    if (!dh1024)
	{
        show_error("%s:DH parameters read failed from %s",func, dh1024_cert_file);
		BIO_free(bio);
		return;
	}
    BIO_free(bio);
}

DH *
PGC_SSL_Server_DH_Callback(SSL *ssl, int is_export, int keylength)
{
    DH *ret;

    if (!dh512 || !dh1024)
	{
        PGC_SSL_Server_Init_DH(  );
	}

    switch (keylength)
    {
        case 512:
            ret = dh512;
            break;
        case 1024:
        default: 
            ret = dh1024;
            break;
    }
    return ret;
}


SSL_CTX *
PGC_SSL_Server_Setup_Ctx(SSL_Server_Info * ssl_server)
{
	static char * func ="PGC_SSL_Server_Setup_Ctx()";
    SSL_CTX *ctx;
	char cafile[PATH_MAX_LENGTH];
 
 	if (ssl_server == NULL)
	{
		show_error("SSL_Server_Info is NULL");
		return NULL;
	}
	DH_Server_Info = ssl_server;

	memset(cafile,0,sizeof(cafile));
	snprintf(cafile,sizeof(cafile),"%s/%s",ssl_server->workPath,CAFILE);

    ctx = SSL_CTX_new(SSLv23_method(  ));
    if (SSL_CTX_load_verify_locations(ctx, cafile, CADIR ) != 1)
	{
        show_error("Error loading CA file and/or directory");
		SSL_CTX_free(ctx);
		return NULL;
	}
    if (SSL_CTX_set_default_verify_paths(ctx) != 1)
	{
        show_error("Error loading default CA file and/or directory");
		SSL_CTX_free(ctx);
		return NULL;
	}
    if (SSL_CTX_use_certificate_chain_file(ctx, ssl_server->serverCertFile) != 1)
	{
        show_error("%s:Error loading certificate from file[%s]",func,ssl_server->serverCertFile);
		SSL_CTX_free(ctx);
		return NULL;
	}
    if (SSL_CTX_use_PrivateKey_file(ctx, ssl_server->serverCertFile, SSL_FILETYPE_PEM) != 1)
	{
        show_error("Error loading private key from file");
		SSL_CTX_free(ctx);
		return NULL;
	}
    SSL_CTX_set_verify(ctx, SSL_VERIFY_PEER|SSL_VERIFY_FAIL_IF_NO_PEER_CERT,
                       PGC_SSL_Verify_Callback);
    SSL_CTX_set_verify_depth(ctx, 4);
    SSL_CTX_set_options(ctx, SSL_OP_ALL | SSL_OP_NO_SSLv2 | SSL_OP_SINGLE_DH_USE);
    SSL_CTX_set_tmp_dh_callback(ctx, PGC_SSL_Server_DH_Callback);
    if (SSL_CTX_set_cipher_list(ctx, CIPHER_LIST) != 1)
	{
        show_error("Error setting cipher list (no valid ciphers)");
		SSL_CTX_free(ctx);
		return NULL;
	}
    return ctx;
}

SSL_CTX *
PGC_SSL_Client_Setup_Ctx(SSL_Server_Info * ssl_server)
{
    SSL_CTX *ctx;
	static char * func="PGC_SSL_Client_Setup_Ctx()";
	char cafile[PATH_MAX_LENGTH];
 
 	if (ssl_server == NULL)
	{
		show_error("%s:SSL_Server_Info is NULL",func);
		return NULL;
	}
	memset(cafile,0,sizeof(cafile));
	snprintf(cafile,sizeof(cafile),"%s/%s",ssl_server->workPath,CAFILE);
    ctx = SSL_CTX_new(SSLv23_method());
    if (SSL_CTX_load_verify_locations(ctx, cafile, CADIR ) != 1)
	{
        show_error("Error loading CA file and/or directory");
		SSL_CTX_free(ctx);
		return NULL;
	}
    if (SSL_CTX_set_default_verify_paths(ctx) != 1)
	{
        show_error("Error loading default CA file and/or directory");
		SSL_CTX_free(ctx);
		return NULL;
	}
    if (SSL_CTX_use_certificate_chain_file(ctx, ssl_server->clientCertFile) != 1)
	{
        show_error("%s:Error loading certificate from file[%s]",func,ssl_server->clientCertFile);
		SSL_CTX_free(ctx);
		return NULL;
	}
    if (SSL_CTX_use_PrivateKey_file(ctx, ssl_server->clientCertFile, SSL_FILETYPE_PEM) != 1)
	{
        show_error("Error loading private key from file");
		SSL_CTX_free(ctx);
		return NULL;
	}
    SSL_CTX_set_verify(ctx, SSL_VERIFY_PEER, PGC_SSL_Verify_Callback);
    SSL_CTX_set_verify_depth(ctx, 4);
    SSL_CTX_set_options(ctx, SSL_OP_ALL|SSL_OP_NO_SSLv2);
    if (SSL_CTX_set_cipher_list(ctx, CIPHER_LIST) != 1)
	{
        show_error("Error setting cipher list (no valid ciphers)");
		SSL_CTX_free(ctx);
		return NULL;
	}
    return ctx;
}

void
PGC_Close_SSL(SSL_Info * ssl_tbl)
{
	PGC_Clear_SSL(ssl_tbl, 1);
}

void
PGC_Clear_SSL(SSL_Info * ssl_tbl,int flag)
{
	/* char * func = "PGC_Clear_SSL()";*/
	if ((ssl_tbl == NULL) || (ssl_tbl->ssl == NULL))
	{
		return;
	}
	if (flag == 0)
	{
		SSL_shutdown(ssl_tbl->ssl);
	}
	else
	{
		SSL_clear(ssl_tbl->ssl);
	}
	SSL_free(ssl_tbl->ssl);
	ssl_tbl->ssl = NULL;
	ERR_remove_state(0);
	if (flag)
	{
		if (ssl_tbl->ctx != NULL)
		{
			SSL_CTX_free(ssl_tbl->ctx);
			ssl_tbl->ctx = NULL;
		}
	}
	return;
}

SSL_Info * 
PGC_Create_Admin_Send_SSL(SSL_Info * ssl_tbl, char * hostName,  uint16_t portNumber)
{
	char * func = "PGC_Create_Admin_Send_SSL()";

	if (AdminTbl == NULL)
	{
		show_error("%s: AdminTbl is null",func);
		return NULL;
	}
	ssl_tbl = create_send_ssl(ssl_tbl, hostName,  portNumber, AdminTbl);
	return ssl_tbl;
}

SSL_Info * 
PGC_Create_Probe_Send_SSL(SSL_Info * ssl_tbl, char * hostName,  uint16_t portNumber)
{
	char * func ="PGC_Create_Probe_Send_SSL()";

	if (ProbeTbl == NULL)
	{
		show_error("%s: ProbeTbl is null",func);	
		return NULL;
	}
	ssl_tbl = create_send_ssl(ssl_tbl, hostName,  portNumber, ProbeTbl);
	return ssl_tbl;
}

static SSL_Info * 
create_send_ssl(SSL_Info * ssl_tbl, char * hostName,  uint16_t portNumber, SSL_Server_Info * ssl_server)
{
	char * func ="create_send_ssl()";	
    long    err;
	char host_port[ MSG_BUF_SIZE];

    PGC_SSL_Init(  );
    PGC_SSL_Seed_Prng(  );
 
    ssl_tbl->ctx = PGC_SSL_Client_Setup_Ctx(ssl_server);
	if(ssl_tbl->ctx == NULL)
	{
		show_error("%s: PGC_SSL_Client_Setup_Ctx failed",func);
		return NULL;
	}
 
	memset(host_port, 0, sizeof(host_port));
 	sprintf(host_port,"%s:%d",hostName,portNumber);
    ssl_tbl->conn = BIO_new_connect(host_port);
    if (!ssl_tbl->conn)
	{
        show_error("%s: BIO_new_connect failed",func);
		ERR_print_errors_fp(stderr);
		SSL_CTX_free(ssl_tbl->ctx);
		ssl_tbl->ctx = NULL;
		return NULL;
	}
    if (BIO_do_connect(ssl_tbl->conn) <= 0)
	{
        show_error("%s: BIO_do_connect failed",func);
		ERR_print_errors_fp(stderr);
		SSL_CTX_free(ssl_tbl->ctx);
		ssl_tbl->ctx = NULL;
		return NULL;
	}
 
    ssl_tbl->ssl = SSL_new(ssl_tbl->ctx);
    SSL_set_bio(ssl_tbl->ssl, ssl_tbl->conn, ssl_tbl->conn);
    if (SSL_connect(ssl_tbl->ssl) <= 0)
	{
        show_error("%s: SSL_connect failed",func);
		ERR_print_errors_fp(stderr);
		SSL_CTX_free(ssl_tbl->ctx);
		ssl_tbl->ctx = NULL;
		return NULL;
	}
    if ((err = PGC_SSL_Post_Connection_Check(ssl_tbl, hostName)) != X509_V_OK)
    {
        show_error("%s: PGC_SSL_Post_Connection_Check failed",func);
		ERR_print_errors_fp(stderr);
		SSL_CTX_free(ssl_tbl->ctx);
		ssl_tbl->ctx = NULL;
		return NULL;
    }
	return ssl_tbl;
}

