/**********************************************
 * Driver.h
 *
 **********************************************/		
#ifndef _H_DRIVER
#define _H_DRIVER

/*****************************************************************************
 * ODBC VERSION (THAT THIS DRIVER COMPLIES WITH)
 *****************************************************************************/
// #define ODBCVER 0x0351

/* Driver stuff */
#define ODBCVER				0x0250
#define DRIVER_ODBC_VER		"02.50"
// #define	ODBCVER				0x0351
// #define DRIVER_ODBC_VER		"03.00"
#define DRIVERNAME			"PostgreSQL ODBC+"
#define DBMS_NAME			"PostgreSQL"
#define POSTGRESDRIVERVERSION	"01.00.00"

#ifdef WIN32
#define DRIVER_FILE_NAME		"PSQLODBCPLUS.DLL"
#define BLCKSZ					4096
#else
#define DRIVER_FILE_NAME		"libpsqlodbcplus.so"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <math.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef WIN32
	#include <sys/types.h>
	#include <sys/socket.h>
	#include <unistd.h>
	#include <netdb.h>
	#include <netinet/in.h>
	#include <arpa/inet.h>
	#include "iodbc.h"
	#include "isql.h"
	#include "isqlext.h"
	#include "misc.h"
	#include "gpps.h"
	
	#if HAVE_PWD_H
		#include <pwd.h>
	#endif

#else
	#include  <windows.h>
	#include  <windowsx.h>
	#include  <odbcinst.h>
	#include <process.h>
	#include "resource.h"
	#include <winsock.h>
	#include <sql.h>
	#include <sqlext.h>
#endif

#include "driverextras.h"

typedef struct ConnectionClass_ ConnectionClass;
typedef struct StatementClass_ StatementClass;
typedef struct QResultClass_ QResultClass;
typedef struct SocketClass_ SocketClass;
typedef struct BindInfoClass_ BindInfoClass;
typedef struct ParameterInfoClass_ ParameterInfoClass;
typedef struct ColumnInfoClass_ ColumnInfoClass;
typedef struct TupleListClass_ TupleListClass;
typedef struct EnvironmentClass_ EnvironmentClass;
typedef struct TupleNode_ TupleNode;
typedef struct TupleField_ TupleField;
typedef struct col_info COL_INFO;
typedef struct lo_arg LO_ARG;

typedef struct GlobalValues_
{
	int					fetch_max;
	int					socket_buffersize;
	int					unknown_sizes;
	int					max_varchar_size;
	int					max_longvarchar_size;
	char				debug;
	char				commlog;
	char				disable_optimizer;
	char				ksqo;
	char				unique_index;
	char				onlyread; /* readonly is reserved on Digital C++ compiler */
	char				use_declarefetch;
	char				text_as_longvarchar;
	char				unknowns_as_longvarchar;
	char				bools_as_char;
	char				lie;
	char				parse;
	char				cancel_as_freestmt;
	char				extra_systable_prefixes[MEDIUM_REGISTRY_LEN];
	char				conn_settings[LARGE_REGISTRY_LEN];
	char				protocol[SMALL_REGISTRY_LEN];

	FILE*				mylogFP;
	FILE*				qlogFP;	
} GLOBAL_VALUES;

typedef struct StatementOptions_ 
{
	int maxRows;
	int maxLength;
	int rowset_size;
	int keyset_size;
	int cursor_type;
	int scroll_concurrency;
	int retrieve_data;
	int bind_size;		        /* size of each structure if using Row Binding */
	int use_bookmarks;
} StatementOptions;

/*	Used to pass extra query info to send_query */
typedef struct QueryInfo_ 
{
	int				row_size;
	QResultClass	*result_in;
	char			*cursor;
} QueryInfo;

GLOBAL_VALUES globals;


void remove_newlines(char *string);
char *strncpy_null(char *dst, const char *src, int len);
char *trim(char *string);
char *make_string(char *s, int len, char *buf);
char *my_strcat(char *buf, char *fmt, char *s, int len);
int my_strcpy(char *dst, int dst_len, char *src, int src_len);

extern GLOBAL_VALUES globals;

/*	This startup packet is to support latest Postgres protocol (6.4, 6.3) */
typedef struct _StartupPacket
{
	ProtocolVersion	protoVersion;
	char			database[SM_DATABASE];
	char			user[SM_USER];
	char			options[SM_OPTIONS];
	char			unused[SM_UNUSED];
	char			tty[SM_TTY];
} StartupPacket;

/*	This startup packet is to support pre-Postgres 6.3 protocol */
typedef struct _StartupPacket6_2
{
	unsigned int	authtype;
	char			database[PATH_SIZE];
	char			user[NAMEDATALEN];
	char			options[ARGV_SIZE];
	char			execfile[ARGV_SIZE];
	char			tty[PATH_SIZE];
} StartupPacket6_2;

/*	Structure to hold all the connection attributes for a specific
	connection (used for both registry and file, DSN and DRIVER)
*/
typedef struct 
{
	char	dsn[MEDIUM_REGISTRY_LEN];
	char	desc[MEDIUM_REGISTRY_LEN];
	char	driver[MEDIUM_REGISTRY_LEN];
	char	server[MEDIUM_REGISTRY_LEN];
	char	database[MEDIUM_REGISTRY_LEN];
	char	username[MEDIUM_REGISTRY_LEN];
	char	password[MEDIUM_REGISTRY_LEN];
	char	conn_settings[LARGE_REGISTRY_LEN];
	char	protocol[SMALL_REGISTRY_LEN];
	char	port[SMALL_REGISTRY_LEN];
	char	onlyread[SMALL_REGISTRY_LEN];	
	char	fake_oid_index[SMALL_REGISTRY_LEN];
	char	show_oid_column[SMALL_REGISTRY_LEN];
	char	row_versioning[SMALL_REGISTRY_LEN];
	char	show_system_tables[SMALL_REGISTRY_LEN];
	char    translation_dll[MEDIUM_REGISTRY_LEN];
	char    translation_option[SMALL_REGISTRY_LEN];
	char	focus_password;
} ConnInfo;

typedef BOOL (FAR WINAPI *DataSourceToDriverProc) (UDWORD,
					SWORD,
					PTR,
					SDWORD,
					PTR,
					SDWORD,
					SDWORD FAR *,
					UCHAR FAR *,
					SWORD,
					SWORD FAR *);

typedef BOOL (FAR WINAPI *DriverToDataSourceProc) (UDWORD,
					SWORD,
					PTR,
					SDWORD,
					PTR,
					SDWORD,
					SDWORD FAR *,
					UCHAR FAR *,
					SWORD,
					SWORD FAR *);

struct EnvironmentClass_ 
{
	char *errormsg;
	int errornumber;
};

/*	Environment prototypes */
EnvironmentClass *EN_Constructor(void);
char EN_Destructor(EnvironmentClass *self);
char EN_get_error(EnvironmentClass *self, int *number, char **message);
char EN_add_connection(EnvironmentClass *self, ConnectionClass *conn);
char EN_remove_connection(EnvironmentClass *self, ConnectionClass *conn);
void EN_log_error(char *func, char *desc, EnvironmentClass *self);

struct ColumnInfoClass_ 
{
	Int2	num_fields;
	char	**name;				/* list of type names */
	Oid		*adtid;				/* list of type ids */
	Int2	*adtsize;			/* list type sizes */
	Int2	*display_size;		/* the display size (longest row) */
	Int4	*atttypmod;			/* the length of bpchar/varchar */
};

struct SocketClass_ 
{

	int buffer_filled_in;
	int buffer_filled_out;
	int buffer_read_in;
	unsigned char *buffer_in;
	unsigned char *buffer_out;

	SOCKETFD socket;

	char *errormsg;
	int errornumber;

	char reverse;	/* used to handle Postgres 6.2 protocol (reverse byte order) */

};

/* Socket prototypes */
SocketClass *SOCK_Constructor(void);
void SOCK_Destructor(SocketClass *self);
char SOCK_connect_to(SocketClass *self, unsigned short port, char *hostname);
void SOCK_get_n_char(SocketClass *self, char *buffer, int len);
void SOCK_put_n_char(SocketClass *self, char *buffer, int len);
void SOCK_get_string(SocketClass *self, char *buffer, int bufsize);
void SOCK_put_string(SocketClass *self, char *string);
int SOCK_get_int(SocketClass *self, short len);
void SOCK_put_int(SocketClass *self, int value, short len);
void SOCK_flush_output(SocketClass *self);
unsigned char SOCK_get_next_byte(SocketClass *self);
void SOCK_put_next_byte(SocketClass *self, unsigned char next_byte);
void SOCK_clear_error(SocketClass *self);

typedef enum 
{
    CONN_NOT_CONNECTED,      /* Connection has not been established */
    CONN_CONNECTED,      /* Connection is up and has been established */
    CONN_DOWN,            /* Connection is broken */
    CONN_EXECUTING     /* the connection is currently executing a statement */
} CONN_Status;

struct ConnectionClass_ 
{
	HENV			henv;					/* environment this connection was created on */
	StatementOptions stmtOptions;
	char			*errormsg;
	int				errornumber;
	CONN_Status		status;
	ConnInfo		connInfo;
	StatementClass	**stmts;
	int				num_stmts;
	SocketClass		*sock;
	int				lobj_type;
	int				ntables;
	COL_INFO		**col_info;
	long            translation_option;
	HINSTANCE       translation_handle;
	DataSourceToDriverProc  DataSourceToDriver;
	DriverToDataSourceProc  DriverToDataSource;
	char			transact_status;		/* Is a transaction is currently in progress */
	char			errormsg_created;		/* has an informative error msg been created?  */
	char			pg_version[MAX_INFO_STRING];	/* Version of PostgreSQL we're connected to - DJP 25-1-2001 */
	float			pg_version_number;
	Int2			pg_version_major;
	Int2			pg_version_minor;
};

/*	prototypes */
ConnectionClass *CC_Constructor(void);
char CC_Destructor(ConnectionClass *self);
int CC_cursor_count(ConnectionClass *self);
char CC_cleanup(ConnectionClass *self);
char CC_abort(ConnectionClass *self);
int CC_set_translation (ConnectionClass *self);
char CC_connect(ConnectionClass *self, char do_password);
char CC_add_statement(ConnectionClass *self, StatementClass *stmt);
char CC_remove_statement(ConnectionClass *self, StatementClass *stmt);
char CC_get_error(ConnectionClass *self, int *number, char **message);
QResultClass *CC_send_query(ConnectionClass *self, char *query, QueryInfo *qi);
void CC_clear_error(ConnectionClass *self);
char *CC_create_errormsg(ConnectionClass *self);
int CC_send_function(ConnectionClass *conn, int fnid, void *result_buf, int *actual_result_len, int result_is_int, LO_ARG *argv, int nargs);
char CC_send_settings(ConnectionClass *self);
void CC_lookup_lo(ConnectionClass *conn);
void CC_lookup_pg_version(ConnectionClass *conn);
void CC_initialize_pg_version(ConnectionClass *conn);
void CC_log_error(char *func, char *desc, ConnectionClass *self);

/*
 * BindInfoClass -- stores information about a bound column
 */
struct BindInfoClass_ 
{
	Int4 buflen;		/* size of buffer */
	Int4 data_left;		/* amount of data left to read (SQLGetData) */
	char *buffer;		/* pointer to the buffer */
	Int4 *used;			/* used space in the buffer (for strings not counting the '\0') */
	Int2 returntype;	/* kind of conversion to be applied when returning (SQL_C_DEFAULT, SQL_C_CHAR...) */
};

/*
 * ParameterInfoClass -- stores information about a bound parameter
 */
struct ParameterInfoClass_ 
{
	Int4 buflen;
	char *buffer;
	Int4 *used;
	Int2 paramType;
	Int2 CType;
	Int2 SQLType;
	UInt4 precision;
	Int2 scale;
	Oid  lobj_oid;
	Int4 *EXEC_used;		/* amount of data OR the oid of the large object */
	char *EXEC_buffer;		/* the data or the FD of the large object */
	char data_at_exec;
};

BindInfoClass *create_empty_bindings(int num_columns);
void extend_bindings(StatementClass *stmt, int num_columns);

typedef enum
{
    STMT_ALLOCATED,     /* The statement handle is allocated, but not used so far */
    STMT_READY,         /* the statement is waiting to be executed */
    STMT_PREMATURE,     /* ODBC states that it is legal to call e.g. SQLDescribeCol before
                           a call to SQLExecute, but after SQLPrepare. To get all the necessary
                           information in such a case, we simply execute the query _before_ the
                           actual call to SQLExecute, so that statement is considered to be "premature".
                        */   
    STMT_FINISHED,      /* statement execution has finished */
    STMT_EXECUTING      /* statement execution is still going on */
} STMT_Status;

/* statement types */
enum 
{
	STMT_TYPE_UNKNOWN = -2,
	STMT_TYPE_OTHER = -1,
	STMT_TYPE_SELECT = 0,
	STMT_TYPE_INSERT,
	STMT_TYPE_UPDATE,
	STMT_TYPE_DELETE,
	STMT_TYPE_CREATE,
	STMT_TYPE_ALTER,
	STMT_TYPE_DROP,
	STMT_TYPE_GRANT,
	STMT_TYPE_REVOKE,
};

/*	Parsing status */
enum 
{
	STMT_PARSE_NONE = 0,
	STMT_PARSE_COMPLETE,
	STMT_PARSE_INCOMPLETE,
	STMT_PARSE_FATAL,
};

/*	Result style */
enum 
{
	STMT_FETCH_NONE = 0,
	STMT_FETCH_NORMAL,
	STMT_FETCH_EXTENDED,
};

typedef struct 
{
	COL_INFO		*col_info;		/* cached SQLColumns info for this table */
	char 			name[MAX_TABLE_LEN+1];
	char			alias[MAX_TABLE_LEN+1];
} TABLE_INFO;

typedef struct 
{
	TABLE_INFO  	*ti;			/* resolve to explicit table names */
	int				precision;
	int				display_size;
	int				length;
	int				type;
	char			nullable;
	char			func;
	char			expr;
	char			quote;
	char			dquote;
	char			numeric;
	char			dot[MAX_TABLE_LEN+1];
	char			name[MAX_COLUMN_LEN+1];
	char			alias[MAX_COLUMN_LEN+1];
} FIELD_INFO;

/********	Statement Handle	***********/
struct StatementClass_ {
    ConnectionClass *hdbc;		/* pointer to ConnectionClass this statement belongs to */
    QResultClass *result;		/* result of the current statement */
	HSTMT FAR *phstmt;
	StatementOptions options;
    STMT_Status status;
    char *errormsg;
    int errornumber;

    /* information on bindings */
    BindInfoClass *bindings;	/* array to store the binding information */
	BindInfoClass bookmark;
    int bindings_allocated;

    /* information on statement parameters */
    int parameters_allocated;
    ParameterInfoClass *parameters;
	Int4 currTuple;				/* current absolute row number (GetData, SetPos, SQLFetch) */
	int  save_rowset_size;		/* saved rowset size in case of change/FETCH_NEXT */
	int  rowset_start;			/* start of rowset (an absolute row number) */
	int  bind_row;				/* current offset for Multiple row/column binding */
	int  last_fetch_count;      /* number of rows retrieved in last fetch/extended fetch */
	int  current_col;			/* current column for GetData -- used to handle multiple calls */
	int  lobj_fd;				/* fd of the current large object */
    char *statement;			/* if non--null pointer to the SQL statement that has been executed */
	TABLE_INFO	**ti;
	FIELD_INFO	**fi;
	int			nfld;
	int			ntab;
	int parse_status;
    int statement_type;			/* According to the defines above */
	int data_at_exec;			/* Number of params needing SQLPutData */
	int current_exec_param;		/* The current parameter for SQLPutData */
	char put_data;				/* Has SQLPutData been called yet? */
	char errormsg_created;		/* has an informative error msg been created?  */
	char manual_result;			/* Is the statement result manually built? */
	char prepare;				/* is this statement a prepared statement or direct */
	char internal;				/* Is this statement being called internally? */
	char cursor_name[MAX_CURSOR_LEN+1];
	char stmt_with_params[STD_STATEMENT_LEN];		/* statement after parameter substitution */
};

/*	Statement prototypes */
StatementClass *SC_Constructor(void);
void InitializeStatementOptions(StatementOptions *opt);
char SC_Destructor(StatementClass *self);
int statement_type(char *statement);
char parse_statement(StatementClass *stmt);
void SC_pre_execute(StatementClass *self);
char SC_unbind_cols(StatementClass *self);
char SC_recycle_statement(StatementClass *self);
void SC_clear_error(StatementClass *self);
char SC_get_error(StatementClass *self, int *number, char **message);
char *SC_create_errormsg(StatementClass *self);
RETCODE SC_execute(StatementClass *self);
RETCODE SC_fetch(StatementClass *self);
void SC_free_params(StatementClass *self, char option);
void SC_log_error(char *func, char *desc, StatementClass *self);
unsigned long SC_get_bookmark(StatementClass *self);

/*	This is used to store cached table information in the connection */
struct col_info 
{
	QResultClass	*result;
	char			name[MAX_TABLE_LEN+1];
};

ColumnInfoClass *CI_Constructor(void);
void CI_Destructor(ColumnInfoClass *self);
void CI_free_memory(ColumnInfoClass *self);
char CI_read_fields(ColumnInfoClass *self, ConnectionClass *conn);

/* functions for setting up the fields from within the program, */
/* without reading from a socket */
void CI_set_num_fields(ColumnInfoClass *self, int new_num_fields);
void CI_set_field_info(ColumnInfoClass *self, int field_num, char *new_name, 
                       Oid new_adtid, Int2 new_adtsize, Int4 atttypmod);

typedef struct 
{
	int m;
	int d;
	int y;
	int hh;
	int mm;
	int ss;
} SIMPLE_TIME;

int copy_and_convert_field_bindinfo(StatementClass *stmt, Int4 field_type, void *value, int col);
int copy_and_convert_field(StatementClass *stmt, Int4 field_type, void *value, Int2 fCType, 
						   PTR rgbValue, SDWORD cbValueMax, SDWORD *pcbValue);
int copy_statement_with_parameters(StatementClass *stmt);
char *convert_escape(char *value);
char *convert_money(char *s);
char parse_datetime(char *buf, SIMPLE_TIME *st);
int convert_linefeeds(char *s, char *dst, size_t max);
char *convert_special_chars(char *si, char *dst, int used);
int convert_pgbinary_to_char(char *value, char *rgbValue, int cbValueMax);
int convert_from_pgbinary(unsigned char *value, unsigned char *rgbValue, int cbValueMax);
int convert_to_pgbinary(unsigned char *in, char *out, int len);
void encode(char *in, char *out);
void decode(char *in, char *out);
int convert_lo(StatementClass *stmt, void *value, Int2 fCType, PTR rgbValue, 
		   SDWORD cbValueMax, SDWORD *pcbValue);

enum QueryResultCode_ 
{
  PGRES_EMPTY_QUERY = 0,
  PGRES_COMMAND_OK,  /* a query command that doesn't return */
                    /* anything was executed properly by the backend */
  PGRES_TUPLES_OK,  /* a query command that returns tuples */
                   /* was executed properly by the backend, PGresult */
                   /* contains the resulttuples */
  PGRES_COPY_OUT,
  PGRES_COPY_IN,
  PGRES_BAD_RESPONSE, /* an unexpected response was recv'd from the backend */
  PGRES_NONFATAL_ERROR,
  PGRES_FATAL_ERROR,
  PGRES_FIELDS_OK,	/* field information from a query was successful */
  PGRES_END_TUPLES,
  PGRES_INTERNAL_ERROR
};
typedef enum QueryResultCode_ QueryResultCode;

struct QResultClass_ 
{
    ColumnInfoClass *fields;			/* the Column information */
    TupleListClass *manual_tuples;		/* manual result tuple list */
	ConnectionClass *conn;				/* the connection this result is using (backend) */

	/*	Stuff for declare/fetch tuples */
	int fetch_count;					/* logical rows read so far */
	int fcount;							/* actual rows read in the fetch */
	int currTuple;
	int base;

	int num_fields;						/* number of fields in the result */
	int cache_size;
	int rowset_size;

    QueryResultCode status;

    char *message;
	char *cursor;						/* The name of the cursor for select statements */
	char *command;
	char *notice;

	TupleField *backend_tuples;			/* data from the backend (the tuple cache) */
	TupleField *tupleField;				/* current backend tuple being retrieved */

	char inTuples;						/* is a fetch of rows from the backend in progress? */
	char aborted;					/* was aborted?*/
};

/*	Core Functions */
QResultClass *QR_Constructor(void);
void QR_Destructor(QResultClass *self);
char QR_read_tuple(QResultClass *self, char binary);
int QR_next_tuple(QResultClass *self);
int QR_close(QResultClass *self);
char QR_fetch_tuples(QResultClass *self, ConnectionClass *conn, char *cursor);
void QR_free_memory(QResultClass *self);
void QR_set_command(QResultClass *self, char *msg);
void QR_set_notice(QResultClass *self, char *msg);
void QR_set_num_fields(QResultClass *self, int new_num_fields); /* manual result only */
void QR_inc_base(QResultClass *self, int base_inc);
void QR_set_cache_size(QResultClass *self, int cache_size);
void QR_set_rowset_size(QResultClass *self, int rowset_size);
void QR_set_position(QResultClass *self, int pos);

Int4 sqltype_to_pgtype(Int2 fSqlType);
Int2 pgtype_to_sqltype(StatementClass *stmt, Int4 type);
Int2 pgtype_to_ctype(StatementClass *stmt, Int4 type);
char *pgtype_to_name(StatementClass *stmt, Int4 type);

/*	These functions can use static numbers or result sets(col parameter) */
Int4 pgtype_precision(StatementClass *stmt, Int4 type, int col, int handle_unknown_size_as);
Int4 pgtype_display_size(StatementClass *stmt, Int4 type, int col, int handle_unknown_size_as);
Int4 pgtype_length(StatementClass *stmt, Int4 type, int col, int handle_unknown_size_as);
Int2 pgtype_scale(StatementClass *stmt, Int4 type, int col);
Int2 pgtype_radix(StatementClass *stmt, Int4 type);
Int2 pgtype_nullable(StatementClass *stmt, Int4 type);
Int2 pgtype_auto_increment(StatementClass *stmt, Int4 type);
Int2 pgtype_case_sensitive(StatementClass *stmt, Int4 type);
Int2 pgtype_money(StatementClass *stmt, Int4 type);
Int2 pgtype_searchable(StatementClass *stmt, Int4 type);
Int2 pgtype_unsigned(StatementClass *stmt, Int4 type);
char *pgtype_literal_prefix(StatementClass *stmt, Int4 type);
char *pgtype_literal_suffix(StatementClass *stmt, Int4 type);
char *pgtype_create_params(StatementClass *stmt, Int4 type);
Int2 sqltype_to_default_ctype(Int2 sqltype);

struct lo_arg 
{
	int		isint;
	int		len;
	union
	{
		int		integer;
		char	*ptr;
	} u;
};


Oid lo_creat(ConnectionClass *conn, int mode);
int lo_open(ConnectionClass *conn, int lobjId, int mode);
int lo_close(ConnectionClass *conn, int fd);
int lo_read(ConnectionClass *conn, int fd, char *buf, int len);
int lo_write(ConnectionClass *conn, int fd, char *buf, int len);
int lo_lseek(ConnectionClass *conn, int fd, int offset, int len);
int lo_tell(ConnectionClass *conn, int fd);
int lo_unlink(ConnectionClass *conn, Oid lobjId);

struct TupleListClass_ 
{
	Int4 num_fields;
	Int4 num_tuples;  
	TupleNode *list_start, *list_end, *lastref;
	Int4 last_indexed;
};

/* Create a TupleList. Each tuple consits of fieldcnt columns */
TupleListClass *TL_Constructor(UInt4 fieldcnt);
void TL_Destructor(TupleListClass *self);  
void *TL_get_fieldval(TupleListClass *self, Int4 tupleno, Int2 fieldno);
char TL_add_tuple(TupleListClass *self, TupleNode *new_field);

/*	Used by backend data AND manual result sets */
struct TupleField_ 
{
    Int4 len;     /* length of the current Tuple */
    void *value;  /* an array representing the value */
};

/*	Used ONLY for manual result sets */
struct TupleNode_ 
{
    struct TupleNode_ *prev, *next;
    TupleField tuple[1];
};

/*	These macros are wrappers for the corresponding set_tuplefield functions
	but these handle automatic NULL determination and call set_tuplefield_null()
	if appropriate for the datatype (used by SQLGetTypeInfo).
*/
void set_tuplefield_null(TupleField *tuple_field);
void set_tuplefield_string(TupleField *tuple_field, char *string);
void set_tuplefield_int2(TupleField *tuple_field, Int2 value);
void set_tuplefield_int4(TupleField *tuple_field, Int4 value);

/*  prototypes */
void getGlobalDefaults(char *section, char *filename, char override);

#ifdef WIN32
void SetDlgStuff(HWND hdlg, ConnInfo *ci);
void GetDlgStuff(HWND hdlg, ConnInfo *ci);
int CALLBACK driver_optionsProc(HWND   hdlg,
                           WORD   wMsg,
                           WPARAM wParam,
                           LPARAM lParam);
int CALLBACK ds_optionsProc(HWND   hdlg,
                           WORD   wMsg,
                           WPARAM wParam,
                           LPARAM lParam);
#endif /* WIN32 */

void updateGlobals(void);
void writeDSNinfo(ConnInfo *ci);
void getDSNdefaults(ConnInfo *ci);
void getDSNinfo(ConnInfo *ci, char overwrite);
void makeConnectString(char *connect_string, ConnInfo *ci);
void copyAttributes(ConnInfo *ci, char *attribute, char *value);

#endif


