/* Module:          SQLExtendedFetch.c
 *
 * Description:     Fetches the specified rowset of data from the result set 
 *					and returns data for all bound columns. 
 *
 * Classes:         
 *
 * API functions:   SQLExtendedFetch
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLExtendedFetch(
								   SQLHSTMT			hDrvStmt,
								   SQLUSMALLINT		nOrientation,
								   SQLINTEGER		nOffset,
								   SQLUINTEGER		*pnRowCount,
								   SQLUSMALLINT		*pRowStatusArray
								   )
{
	static char *func = "SQLExtendedFetch";
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	QResultClass *res;
	int num_tuples, i, save_rowset_size;
	SQLRETURN result;
	char truncated, error;

	mylog("SQLExtendedFetch: stmt=%u\n", stmt);

	if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	if ( globals.use_declarefetch && ! stmt->manual_result)
	{
		if ( nOrientation != SQL_FETCH_NEXT)
		{
			stmt->errornumber = STMT_NOT_IMPLEMENTED_ERROR;
			stmt->errormsg = "Unsupported fetch type for SQLExtendedFetch with UseDeclareFetch option.";
			return SQL_ERROR;
		}
	}

	SC_clear_error(stmt);

	if ( ! (res = stmt->result))
	{
		stmt->errormsg = "Null statement result in SQLExtendedFetch.";
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	/*	If a bookmark colunmn is bound but bookmark usage is off, then error */
	if (stmt->bookmark.buffer && stmt->options.use_bookmarks == SQL_UB_OFF)
	{
		stmt->errornumber = STMT_COLNUM_ERROR;
		stmt->errormsg = "Attempt to retrieve bookmark with bookmark usage disabled";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	if (stmt->status == STMT_EXECUTING)
	{
		stmt->errormsg = "Can't fetch while statement is still executing.";
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	if (stmt->status != STMT_FINISHED)
	{
		stmt->errornumber = STMT_STATUS_ERROR;
		stmt->errormsg = "ExtendedFetch can only be called after the successful execution on a SQL statement";
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	if (stmt->bindings == NULL) 
	{
		/* just to avoid a crash if the user insists on calling this */
		/* function even if SQL_ExecDirect has reported an Error */
		stmt->errormsg = "Bindings were not allocated properly.";
		stmt->errornumber = STMT_SEQUENCE_ERROR;
		SC_log_error(func, "", stmt);
		return SQL_ERROR;
	}

	/*	Initialize to no rows fetched */
	if (pRowStatusArray)
	{
		for (i = 0; i < stmt->options.rowset_size; i++)
		{
			*(pRowStatusArray + i) = SQL_ROW_NOROW;
		}
	}

	if (pnRowCount)
	{
		*pnRowCount = 0;
	}

	num_tuples = QR_get_num_tuples(res);

	/*	Save and discard the saved rowset size */
	save_rowset_size = stmt->save_rowset_size;
	stmt->save_rowset_size = -1;

	switch (nOrientation)
	{
		case SQL_FETCH_NEXT:
			/*	From the odbc spec... If positioned before the start of the RESULT SET,
			then this should be equivalent to SQL_FETCH_FIRST.
			*/
			if (stmt->rowset_start < 0)
			{
				stmt->rowset_start = 0;
			}
			else
			{
				stmt->rowset_start += (save_rowset_size > 0 ? save_rowset_size : stmt->options.rowset_size);
			}

			mylog("SQL_FETCH_NEXT: num_tuples=%d, currtuple=%d\n", num_tuples, stmt->currTuple);
			break;

		case SQL_FETCH_PRIOR:
			mylog("SQL_FETCH_PRIOR: num_tuples=%d, currtuple=%d\n", num_tuples, stmt->currTuple);

			/*	From the odbc spec... If positioned after the end of the RESULT SET,
			then this should be equivalent to SQL_FETCH_LAST.
			*/

			if (stmt->rowset_start >= num_tuples)
			{
			stmt->rowset_start = num_tuples <= 0 ? 0 : (num_tuples - stmt->options.rowset_size);
			}
			else 
			{

			stmt->rowset_start -= stmt->options.rowset_size;
			}
			break;

		case SQL_FETCH_FIRST:
			mylog("SQL_FETCH_FIRST: num_tuples=%d, currtuple=%d\n", num_tuples, stmt->currTuple);
			stmt->rowset_start = 0;
			break;

		case SQL_FETCH_LAST:
			mylog("SQL_FETCH_LAST: num_tuples=%d, currtuple=%d\n", num_tuples, stmt->currTuple);
			stmt->rowset_start = num_tuples <= 0 ? 0 : (num_tuples - stmt->options.rowset_size) ;
			break;


		case SQL_FETCH_ABSOLUTE:
			mylog("SQL_FETCH_ABSOLUTE: num_tuples=%d, currtuple=%d, nOffset=%d\n", num_tuples, stmt->currTuple, nOffset);

			/*	Position before result set, but dont fetch anything */
			if (nOffset == 0)
			{
				stmt->rowset_start = -1;
				stmt->currTuple = -1;
				return SQL_NO_DATA_FOUND;
			}
			/*	Position before the desired row */
			else if (nOffset > 0)
			{
			stmt->rowset_start = nOffset - 1;
			}
			/*	Position with respect to the end of the result set */
			else 
			{
				stmt->rowset_start = num_tuples + nOffset;
			}  
			break;

		case SQL_FETCH_RELATIVE:

			/*	Refresh the current rowset -- not currently implemented, but lie anyway */
			if (nOffset == 0)
			{
				break;
			}
			stmt->rowset_start += nOffset;
			break;

		case SQL_FETCH_BOOKMARK:
			stmt->rowset_start = nOffset - 1;
			break;
		
		default:
			SC_log_error(func, "Unsupported SQLExtendedFetch Direction", stmt);
			return SQL_ERROR; 
	}           


	/***********************************/
	/*	CHECK FOR PROPER CURSOR STATE  */
	/***********************************/
	/*	Handle Declare Fetch style specially because the end is not really the end... */
	if ( globals.use_declarefetch && ! stmt->manual_result)
	{
		if (QR_end_tuples(res))
		{
			return SQL_NO_DATA_FOUND;
		}
	}
	else
	{
		/*	If *new* rowset is after the result_set, return no data found */
		if (stmt->rowset_start >= num_tuples) 
		{
			stmt->rowset_start = num_tuples;
			return SQL_NO_DATA_FOUND;
		}
	}

	/*	If *new* rowset is prior to result_set, return no data found */
	if (stmt->rowset_start < 0)
	{
		if (stmt->rowset_start + stmt->options.rowset_size <= 0)
		{
			stmt->rowset_start = -1;
			return SQL_NO_DATA_FOUND;
		}
		else 
		{	/*	overlap with beginning of result set, so get first rowset */
			stmt->rowset_start = 0;
		}
	}

	/*	currTuple is always 1 row prior to the rowset */
	stmt->currTuple = stmt->rowset_start - 1;

	/*	increment the base row in the tuple cache */
	QR_set_rowset_size(res, stmt->options.rowset_size);
	QR_inc_base(res, stmt->last_fetch_count);	
		
	/*	Physical Row advancement occurs for each row fetched below */

	mylog("SQLExtendedFetch: new currTuple = %d\n", stmt->currTuple);

	truncated = error = FALSE;
	for (i = 0; i < stmt->options.rowset_size; i++) 
	{

		stmt->bind_row = i;		/* set the binding location */
		result = SC_fetch(stmt);

		/*	Determine Function status */
		if (result == SQL_NO_DATA_FOUND)
		{
			break;
		}
		else if (result == SQL_SUCCESS_WITH_INFO)
		{
			truncated = TRUE;
		}
		else if (result == SQL_ERROR)
		{
			error = TRUE;
		}

		/*	Determine Row Status */
		if (pRowStatusArray) 
		{
			if (result == SQL_ERROR) 
			{
				*(pRowStatusArray + i) = SQL_ROW_ERROR;
			}
			else
			{
				*(pRowStatusArray + i)= SQL_ROW_SUCCESS;
			}
		}
	}

	/*	Save the fetch count for SQLSetPos */
	stmt->last_fetch_count= i;

	/*	Reset next binding row */
	stmt->bind_row = 0;

	/*	Move the cursor position to the first row in the result set. */
	stmt->currTuple = stmt->rowset_start;

	/*	For declare/fetch, need to reset cursor to beginning of rowset */
	if (globals.use_declarefetch && ! stmt->manual_result) 
	{
		QR_set_position(res, 0);
	}

	/*	Set the number of rows retrieved */
	if (pnRowCount)
	{
		*pnRowCount = i;
	}

	if (i == 0)
	{
		return SQL_NO_DATA_FOUND;	
	}
	/*	Only DeclareFetch should wind up here */
	else if (error)
	{
		return SQL_ERROR;
	}
	else if (truncated)
	{
		return SQL_SUCCESS_WITH_INFO;
	}
	else
	{
		return SQL_SUCCESS;
	}

}


